/*
 * Copyright (c) 2019-2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __aarch64__

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void a64_hybrid_s8qa_dot_4x16 (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int
)
{
    struct KernelArgs {
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->c_offset > qp->minval) {
        flags |= 0x20;
    }
    __asm__ __volatile__(
      "1:"  // Row loop
      "cmp %x[M], #0x4\n"
      "bge 91f\n"
      "cmp %x[M], #0x2\n"
      "bgt 61f\n"
      "beq 31f\n"
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "3:"  // Height 1: setup done
      "mov x26, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "cbnz x26, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x24, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x25, #0x10\n"
      "blt 11f\n"
      "ldr q0, [x24, #0x0]\n"
      "ldr q4, [x28, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "blt 9f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "ldr q21, [x28, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "ldr q20, [x28, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "ldr q26, [x28, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      "ldr q25, [x28, #0xa0]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      "ldr q24, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      "ldr q23, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      "ldr q22, [x28, #0xd0]\n"
      ".inst 0x4fa0e2b3  // sdot v19.4s, v21.16b, v0.4b[1]\n"
      "ldr q21, [x28, #0xe0]\n"
      ".inst 0x4f80ea90  // sdot v16.4s, v20.16b, v0.4b[2]\n"
      "ldr q20, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f80eb51  // sdot v17.4s, v26.16b, v0.4b[2]\n"
      ".inst 0x4f80eb32  // sdot v18.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f80eb13  // sdot v19.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4fa0eaf0  // sdot v16.4s, v23.16b, v0.4b[3]\n"
      ".inst 0x4fa0ead1  // sdot v17.4s, v22.16b, v0.4b[3]\n"
      ".inst 0x4fa0eab2  // sdot v18.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4fa0ea93  // sdot v19.4s, v20.16b, v0.4b[3]\n"
      "tbnz %x[flags], #31, 8f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "8:"  // Height 1: Multiply loop: unique 1: skip row sum
      "ldr q0, [x24, #0x0]\n"
      "ldr q4, [x28, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "cmp x25, #0x20\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "bge 7b\n"
      "9:"  // Height 1: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      "ldr q21, [x28, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "ldr q20, [x28, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      "ldr q26, [x28, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      "ldr q25, [x28, #0xa0]\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      "ldr q24, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      "ldr q23, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      "ldr q22, [x28, #0xd0]\n"
      ".inst 0x4fa0e2b3  // sdot v19.4s, v21.16b, v0.4b[1]\n"
      "ldr q21, [x28, #0xe0]\n"
      ".inst 0x4f80ea90  // sdot v16.4s, v20.16b, v0.4b[2]\n"
      "ldr q20, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f80eb51  // sdot v17.4s, v26.16b, v0.4b[2]\n"
      ".inst 0x4f80eb32  // sdot v18.4s, v25.16b, v0.4b[2]\n"
      ".inst 0x4f80eb13  // sdot v19.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4fa0eaf0  // sdot v16.4s, v23.16b, v0.4b[3]\n"
      ".inst 0x4fa0ead1  // sdot v17.4s, v22.16b, v0.4b[3]\n"
      ".inst 0x4fa0eab2  // sdot v18.4s, v21.16b, v0.4b[3]\n"
      ".inst 0x4fa0ea93  // sdot v19.4s, v20.16b, v0.4b[3]\n"
      "tbnz %x[flags], #31, 10f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "10:"  // Height 1: Multiply loop: unique 2: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "11:"  // Height 1: Multiply loop: Main loop skip
      "cbz x25, 18f\n"
      "cmp x25, #0x4\n"
      "blt 14f\n"
      "12:"  // Height 1: Multiply loop: Odd block loop
      "ldr s0, [x24], #0x4\n"
      "tbnz %x[flags], #31, 13f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "13:"  // Height 1: Multiply loop: unique 3: skip row sum
      "ldr q23, [x28, #0x0]\n"
      "ldr q22, [x28, #0x10]\n"
      "sub x25, x25, #0x4\n"
      "ldr q21, [x28, #0x20]\n"
      "ldr q20, [x28, #0x30]\n"
      "cmp x25, #0x4\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e2f0  // sdot v16.4s, v23.16b, v0.4b[0]\n"
      ".inst 0x4f80e2d1  // sdot v17.4s, v22.16b, v0.4b[0]\n"
      ".inst 0x4f80e2b2  // sdot v18.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f80e293  // sdot v19.4s, v20.16b, v0.4b[0]\n"
      "bge 12b\n"
      "14:"  // Height 1: Multiply loop: Skip odd blocks
      "cbz x25, 18f\n"
      "tbz x25, #1, 15f\n"
      "ldr h0, [x24], #0x2\n"
      "tbz x25, #0, 16f\n"
      "ld1 { v0.b }[2], [x24]\n"
      "b 16f\n"
      "15:"  // Height 1: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x24, #0x0]\n"
      "16:"  // Height 1: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 17f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      "17:"  // Height 1: Multiply loop: unique 4: skip row sum
      "ldr q23, [x28, #0x0]\n"
      "ldr q22, [x28, #0x10]\n"
      "ldr q21, [x28, #0x20]\n"
      "ldr q20, [x28, #0x30]\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e2f0  // sdot v16.4s, v23.16b, v0.4b[0]\n"
      ".inst 0x4f80e2d1  // sdot v17.4s, v22.16b, v0.4b[0]\n"
      ".inst 0x4f80e2b2  // sdot v18.4s, v21.16b, v0.4b[0]\n"
      ".inst 0x4f80e293  // sdot v19.4s, v20.16b, v0.4b[0]\n"
      "18:"  // Height 1: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 4b\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "tbnz %x[flags], #31, 19f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "ld1r { v20.4s }, [x20]\n"
      "neg v20.4s, v20.4s\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "mul v11.4s, v11.4s, v20.4s\n"
      "19:"  // Height 1: skip row sum fixup
      "ldr q24, [x10, #0x0]\n"
      "ldr q23, [x10, #0x10]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q22, [x10, #0x20]\n"
      "ldr q21, [x10, #0x30]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v20.4s }, [x20]\n"
      "add v16.4s, v16.4s, v24.4s\n"
      "add v17.4s, v17.4s, v23.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add x10, x10, #0x40\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v18.4s, v18.4s, v22.4s\n"
      "add v19.4s, v19.4s, v21.4s\n"
      "sqrdmulh v16.4s, v16.4s, v20.4s\n"
      "sqrdmulh v17.4s, v17.4s, v20.4s\n"
      "sqrdmulh v18.4s, v18.4s, v20.4s\n"
      "sqrdmulh v19.4s, v19.4s, v20.4s\n"
      "tbz %x[flags], #5, 20f\n"
      "and v23.16b, v16.16b, v0.16b\n"
      "and v22.16b, v17.16b, v0.16b\n"
      "and v21.16b, v18.16b, v0.16b\n"
      "and v20.16b, v19.16b, v0.16b\n"
      "sshr v23.4s, v23.4s, #0x1f\n"
      "sshr v22.4s, v22.4s, #0x1f\n"
      "sshr v21.4s, v21.4s, #0x1f\n"
      "sshr v20.4s, v20.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v23.4s\n"
      "sqadd v17.4s, v17.4s, v22.4s\n"
      "sqadd v18.4s, v18.4s, v21.4s\n"
      "sqadd v19.4s, v19.4s, v20.4s\n"
      "20:"  // Height 1: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v22.4s }, [x21]\n"
      "ld1r { v21.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v20.4s }, [x20]\n"
      "add v16.4s, v16.4s, v22.4s\n"
      "add v17.4s, v17.4s, v22.4s\n"
      "add v18.4s, v18.4s, v22.4s\n"
      "add v19.4s, v19.4s, v22.4s\n"
      "smin v16.4s, v16.4s, v21.4s\n"
      "smin v17.4s, v17.4s, v21.4s\n"
      "smin v18.4s, v18.4s, v21.4s\n"
      "smin v19.4s, v19.4s, v21.4s\n"
      "smax v16.4s, v16.4s, v20.4s\n"
      "smax v17.4s, v17.4s, v20.4s\n"
      "smax v18.4s, v18.4s, v20.4s\n"
      "smax v19.4s, v19.4s, v20.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v17.8h, v18.8h, v19.8h\n"
      "uzp1 v16.16b, v16.16b, v17.16b\n"
      "bge 29f\n"
      "tbz x9, #3, 24f\n"
      "str d16, [x27], #0x8\n"
      "tbz x9, #2, 22f\n"
      "st1 { v16.s }[2], [x27], #0x4\n"
      "tbz x9, #1, 21f\n"
      "st1 { v16.h }[6], [x27], #0x2\n"
      "tbz x9, #0, 28f\n"
      "st1 { v16.b }[14], [x27]\n"
      "b 28f\n"
      "21:"  // Height 1: Partial direct writeback: partial_1_12
      "tbz x9, #0, 28f\n"
      "st1 { v16.b }[12], [x27]\n"
      "b 28f\n"
      "22:"  // Height 1: Partial direct writeback: partial_2_8
      "tbz x9, #1, 23f\n"
      "st1 { v16.h }[4], [x27], #0x2\n"
      "tbz x9, #0, 28f\n"
      "st1 { v16.b }[10], [x27]\n"
      "b 28f\n"
      "23:"  // Height 1: Partial direct writeback: partial_1_8
      "tbz x9, #0, 28f\n"
      "st1 { v16.b }[8], [x27]\n"
      "b 28f\n"
      "24:"  // Height 1: Partial direct writeback: partial_4_0
      "tbz x9, #2, 26f\n"
      "str s16, [x27], #0x4\n"
      "tbz x9, #1, 25f\n"
      "st1 { v16.h }[2], [x27], #0x2\n"
      "tbz x9, #0, 28f\n"
      "st1 { v16.b }[6], [x27]\n"
      "b 28f\n"
      "25:"  // Height 1: Partial direct writeback: partial_1_4
      "tbz x9, #0, 28f\n"
      "st1 { v16.b }[4], [x27]\n"
      "b 28f\n"
      "26:"  // Height 1: Partial direct writeback: partial_2_0
      "tbz x9, #1, 27f\n"
      "str h16, [x27], #0x2\n"
      "tbz x9, #0, 28f\n"
      "st1 { v16.b }[2], [x27]\n"
      "b 28f\n"
      "27:"  // Height 1: Partial direct writeback: partial_1_0
      "str b16, [x27, #0x0]\n"
      "28:"  // Height 1: Partial direct writeback: Done
      "b 30f\n"
      "29:"  // Height 1: Full writeback
      "str q16, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "30:"  // Height 1: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 2b\n"
      "b 122f\n"
      "31:"  // Height 2
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v15.16b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "32:"  // Height 2: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "33:"  // Height 2: setup done
      "mov x26, #0x0\n"
      "34:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 35f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "cbnz x26, 36f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 36f\n"
      "35:"  // Height 2: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "36:"  // Height 2: input setup done
      "cmp x25, #0x10\n"
      "blt 41f\n"
      "ldr q0, [x24, #0x0]\n"
      "ldr q1, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "blt 39f\n"
      "37:"  // Height 2: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "ldr q25, [x28, #0x70]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "ldr q24, [x28, #0x80]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      "ldr q30, [x28, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      "ldr q29, [x28, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      "ldr q28, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      "ldr q27, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      "ldr q26, [x28, #0xd0]\n"
      ".inst 0x4fa0e333  // sdot v19.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4fa1e337  // sdot v23.4s, v25.16b, v1.4b[1]\n"
      "ldr q25, [x28, #0xe0]\n"
      ".inst 0x4f80eb10  // sdot v16.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f81eb14  // sdot v20.4s, v24.16b, v1.4b[2]\n"
      "ldr q24, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f80ebd1  // sdot v17.4s, v30.16b, v0.4b[2]\n"
      ".inst 0x4f81ebd5  // sdot v21.4s, v30.16b, v1.4b[2]\n"
      ".inst 0x4f80ebb2  // sdot v18.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f81ebb6  // sdot v22.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f80eb93  // sdot v19.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb97  // sdot v23.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4fa0eb70  // sdot v16.4s, v27.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb74  // sdot v20.4s, v27.16b, v1.4b[3]\n"
      ".inst 0x4fa0eb51  // sdot v17.4s, v26.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb55  // sdot v21.4s, v26.16b, v1.4b[3]\n"
      ".inst 0x4fa0eb32  // sdot v18.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb36  // sdot v22.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4fa0eb13  // sdot v19.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb17  // sdot v23.4s, v24.16b, v1.4b[3]\n"
      "tbnz %x[flags], #31, 38f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "38:"  // Height 2: Multiply loop: unique 5: skip row sum
      "ldr q0, [x24, #0x0]\n"
      "ldr q1, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "cmp x25, #0x20\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "bge 37b\n"
      "39:"  // Height 2: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "ldr q25, [x28, #0x70]\n"
      "sub x25, x25, #0x10\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "ldr q24, [x28, #0x80]\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      "ldr q30, [x28, #0x90]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      "ldr q29, [x28, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      "ldr q28, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      "ldr q27, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      "ldr q26, [x28, #0xd0]\n"
      ".inst 0x4fa0e333  // sdot v19.4s, v25.16b, v0.4b[1]\n"
      ".inst 0x4fa1e337  // sdot v23.4s, v25.16b, v1.4b[1]\n"
      "ldr q25, [x28, #0xe0]\n"
      ".inst 0x4f80eb10  // sdot v16.4s, v24.16b, v0.4b[2]\n"
      ".inst 0x4f81eb14  // sdot v20.4s, v24.16b, v1.4b[2]\n"
      "ldr q24, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f80ebd1  // sdot v17.4s, v30.16b, v0.4b[2]\n"
      ".inst 0x4f81ebd5  // sdot v21.4s, v30.16b, v1.4b[2]\n"
      ".inst 0x4f80ebb2  // sdot v18.4s, v29.16b, v0.4b[2]\n"
      ".inst 0x4f81ebb6  // sdot v22.4s, v29.16b, v1.4b[2]\n"
      ".inst 0x4f80eb93  // sdot v19.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb97  // sdot v23.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4fa0eb70  // sdot v16.4s, v27.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb74  // sdot v20.4s, v27.16b, v1.4b[3]\n"
      ".inst 0x4fa0eb51  // sdot v17.4s, v26.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb55  // sdot v21.4s, v26.16b, v1.4b[3]\n"
      ".inst 0x4fa0eb32  // sdot v18.4s, v25.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb36  // sdot v22.4s, v25.16b, v1.4b[3]\n"
      ".inst 0x4fa0eb13  // sdot v19.4s, v24.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb17  // sdot v23.4s, v24.16b, v1.4b[3]\n"
      "tbnz %x[flags], #31, 40f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "40:"  // Height 2: Multiply loop: unique 6: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "41:"  // Height 2: Multiply loop: Main loop skip
      "cbz x25, 48f\n"
      "cmp x25, #0x4\n"
      "blt 44f\n"
      "42:"  // Height 2: Multiply loop: Odd block loop
      "ldr s0, [x24], #0x4\n"
      "ldr s1, [x23], #0x4\n"
      "tbnz %x[flags], #31, 43f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "43:"  // Height 2: Multiply loop: unique 7: skip row sum
      "ldr q27, [x28, #0x0]\n"
      "ldr q26, [x28, #0x10]\n"
      "sub x25, x25, #0x4\n"
      "ldr q25, [x28, #0x20]\n"
      "ldr q24, [x28, #0x30]\n"
      "cmp x25, #0x4\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e370  // sdot v16.4s, v27.16b, v0.4b[0]\n"
      ".inst 0x4f81e374  // sdot v20.4s, v27.16b, v1.4b[0]\n"
      ".inst 0x4f80e351  // sdot v17.4s, v26.16b, v0.4b[0]\n"
      ".inst 0x4f81e355  // sdot v21.4s, v26.16b, v1.4b[0]\n"
      ".inst 0x4f80e332  // sdot v18.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f81e336  // sdot v22.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x4f80e313  // sdot v19.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f81e317  // sdot v23.4s, v24.16b, v1.4b[0]\n"
      "bge 42b\n"
      "44:"  // Height 2: Multiply loop: Skip odd blocks
      "cbz x25, 48f\n"
      "tbz x25, #1, 45f\n"
      "ldr h0, [x24], #0x2\n"
      "ldr h1, [x23], #0x2\n"
      "tbz x25, #0, 46f\n"
      "ld1 { v0.b }[2], [x24]\n"
      "ld1 { v1.b }[2], [x23]\n"
      "b 46f\n"
      "45:"  // Height 2: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x24, #0x0]\n"
      "ldr b1, [x23, #0x0]\n"
      "46:"  // Height 2: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 47f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      "47:"  // Height 2: Multiply loop: unique 8: skip row sum
      "ldr q27, [x28, #0x0]\n"
      "ldr q26, [x28, #0x10]\n"
      "ldr q25, [x28, #0x20]\n"
      "ldr q24, [x28, #0x30]\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e370  // sdot v16.4s, v27.16b, v0.4b[0]\n"
      ".inst 0x4f81e374  // sdot v20.4s, v27.16b, v1.4b[0]\n"
      ".inst 0x4f80e351  // sdot v17.4s, v26.16b, v0.4b[0]\n"
      ".inst 0x4f81e355  // sdot v21.4s, v26.16b, v1.4b[0]\n"
      ".inst 0x4f80e332  // sdot v18.4s, v25.16b, v0.4b[0]\n"
      ".inst 0x4f81e336  // sdot v22.4s, v25.16b, v1.4b[0]\n"
      ".inst 0x4f80e313  // sdot v19.4s, v24.16b, v0.4b[0]\n"
      ".inst 0x4f81e317  // sdot v23.4s, v24.16b, v1.4b[0]\n"
      "48:"  // Height 2: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 34b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "add x24, x27, x20\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "tbnz %x[flags], #31, 49f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "ld1r { v24.4s }, [x20]\n"
      "neg v24.4s, v24.4s\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "mul v11.4s, v11.4s, v24.4s\n"
      "mul v12.4s, v12.4s, v24.4s\n"
      "49:"  // Height 2: skip row sum fixup
      "ldr q28, [x10, #0x0]\n"
      "ldr q27, [x10, #0x10]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q26, [x10, #0x20]\n"
      "ldr q25, [x10, #0x30]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add v20.4s, v20.4s, v12.4s\n"
      "add v21.4s, v21.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v24.4s }, [x20]\n"
      "add v22.4s, v22.4s, v12.4s\n"
      "add v23.4s, v23.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add v16.4s, v16.4s, v28.4s\n"
      "add v17.4s, v17.4s, v27.4s\n"
      "add x10, x10, #0x40\n"
      "add v18.4s, v18.4s, v26.4s\n"
      "add v19.4s, v19.4s, v25.4s\n"
      "add v20.4s, v20.4s, v28.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v21.4s, v21.4s, v27.4s\n"
      "add v22.4s, v22.4s, v26.4s\n"
      "add v23.4s, v23.4s, v25.4s\n"
      "sqrdmulh v16.4s, v16.4s, v24.4s\n"
      "sqrdmulh v17.4s, v17.4s, v24.4s\n"
      "sqrdmulh v18.4s, v18.4s, v24.4s\n"
      "sqrdmulh v19.4s, v19.4s, v24.4s\n"
      "sqrdmulh v20.4s, v20.4s, v24.4s\n"
      "sqrdmulh v21.4s, v21.4s, v24.4s\n"
      "sqrdmulh v22.4s, v22.4s, v24.4s\n"
      "sqrdmulh v23.4s, v23.4s, v24.4s\n"
      "tbz %x[flags], #5, 50f\n"
      "and v24.16b, v16.16b, v0.16b\n"
      "and v30.16b, v17.16b, v0.16b\n"
      "and v29.16b, v18.16b, v0.16b\n"
      "and v28.16b, v19.16b, v0.16b\n"
      "and v27.16b, v20.16b, v0.16b\n"
      "and v26.16b, v21.16b, v0.16b\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "and v25.16b, v22.16b, v0.16b\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sshr v27.4s, v27.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v24.4s\n"
      "and v24.16b, v23.16b, v0.16b\n"
      "sshr v26.4s, v26.4s, #0x1f\n"
      "sshr v25.4s, v25.4s, #0x1f\n"
      "sqadd v17.4s, v17.4s, v30.4s\n"
      "sqadd v18.4s, v18.4s, v29.4s\n"
      "sshr v24.4s, v24.4s, #0x1f\n"
      "sqadd v19.4s, v19.4s, v28.4s\n"
      "sqadd v20.4s, v20.4s, v27.4s\n"
      "sqadd v21.4s, v21.4s, v26.4s\n"
      "sqadd v22.4s, v22.4s, v25.4s\n"
      "sqadd v23.4s, v23.4s, v24.4s\n"
      "50:"  // Height 2: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v26.4s }, [x21]\n"
      "ld1r { v25.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v24.4s }, [x20]\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "add v16.4s, v16.4s, v26.4s\n"
      "add v17.4s, v17.4s, v26.4s\n"
      "add v18.4s, v18.4s, v26.4s\n"
      "add v19.4s, v19.4s, v26.4s\n"
      "add v20.4s, v20.4s, v26.4s\n"
      "add v21.4s, v21.4s, v26.4s\n"
      "add v22.4s, v22.4s, v26.4s\n"
      "add v23.4s, v23.4s, v26.4s\n"
      "smin v16.4s, v16.4s, v25.4s\n"
      "smin v17.4s, v17.4s, v25.4s\n"
      "smin v18.4s, v18.4s, v25.4s\n"
      "smin v19.4s, v19.4s, v25.4s\n"
      "smin v20.4s, v20.4s, v25.4s\n"
      "smin v21.4s, v21.4s, v25.4s\n"
      "smin v22.4s, v22.4s, v25.4s\n"
      "smin v23.4s, v23.4s, v25.4s\n"
      "smax v16.4s, v16.4s, v24.4s\n"
      "smax v17.4s, v17.4s, v24.4s\n"
      "smax v18.4s, v18.4s, v24.4s\n"
      "smax v19.4s, v19.4s, v24.4s\n"
      "smax v20.4s, v20.4s, v24.4s\n"
      "smax v21.4s, v21.4s, v24.4s\n"
      "smax v22.4s, v22.4s, v24.4s\n"
      "smax v23.4s, v23.4s, v24.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v18.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v17.8h, v22.8h, v23.8h\n"
      "uzp1 v16.16b, v16.16b, v18.16b\n"
      "uzp1 v20.16b, v20.16b, v17.16b\n"
      "bge 59f\n"
      "tbz x9, #3, 54f\n"
      "str d16, [x27], #0x8\n"
      "str d20, [x24], #0x8\n"
      "tbz x9, #2, 52f\n"
      "st1 { v16.s }[2], [x27], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "tbz x9, #1, 51f\n"
      "st1 { v16.h }[6], [x27], #0x2\n"
      "st1 { v20.h }[6], [x24], #0x2\n"
      "tbz x9, #0, 58f\n"
      "st1 { v16.b }[14], [x27]\n"
      "st1 { v20.b }[14], [x24]\n"
      "b 58f\n"
      "51:"  // Height 2: Partial direct writeback: partial_1_12
      "tbz x9, #0, 58f\n"
      "st1 { v16.b }[12], [x27]\n"
      "st1 { v20.b }[12], [x24]\n"
      "b 58f\n"
      "52:"  // Height 2: Partial direct writeback: partial_2_8
      "tbz x9, #1, 53f\n"
      "st1 { v16.h }[4], [x27], #0x2\n"
      "st1 { v20.h }[4], [x24], #0x2\n"
      "tbz x9, #0, 58f\n"
      "st1 { v16.b }[10], [x27]\n"
      "st1 { v20.b }[10], [x24]\n"
      "b 58f\n"
      "53:"  // Height 2: Partial direct writeback: partial_1_8
      "tbz x9, #0, 58f\n"
      "st1 { v16.b }[8], [x27]\n"
      "st1 { v20.b }[8], [x24]\n"
      "b 58f\n"
      "54:"  // Height 2: Partial direct writeback: partial_4_0
      "tbz x9, #2, 56f\n"
      "str s16, [x27], #0x4\n"
      "str s20, [x24], #0x4\n"
      "tbz x9, #1, 55f\n"
      "st1 { v16.h }[2], [x27], #0x2\n"
      "st1 { v20.h }[2], [x24], #0x2\n"
      "tbz x9, #0, 58f\n"
      "st1 { v16.b }[6], [x27]\n"
      "st1 { v20.b }[6], [x24]\n"
      "b 58f\n"
      "55:"  // Height 2: Partial direct writeback: partial_1_4
      "tbz x9, #0, 58f\n"
      "st1 { v16.b }[4], [x27]\n"
      "st1 { v20.b }[4], [x24]\n"
      "b 58f\n"
      "56:"  // Height 2: Partial direct writeback: partial_2_0
      "tbz x9, #1, 57f\n"
      "str h16, [x27], #0x2\n"
      "str h20, [x24], #0x2\n"
      "tbz x9, #0, 58f\n"
      "st1 { v16.b }[2], [x27]\n"
      "st1 { v20.b }[2], [x24]\n"
      "b 58f\n"
      "57:"  // Height 2: Partial direct writeback: partial_1_0
      "str b16, [x27, #0x0]\n"
      "str b20, [x24, #0x0]\n"
      "58:"  // Height 2: Partial direct writeback: Done
      "b 60f\n"
      "59:"  // Height 2: Full writeback
      "str q16, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q20, [x24, #0x0]\n"
      "60:"  // Height 2: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 32b\n"
      "b 122f\n"
      "61:"  // Height 3
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "movi v13.4s, #0x0\n"
      "movi v15.16b, #0x1\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "62:"  // Height 3: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "63:"  // Height 3: setup done
      "mov x26, #0x0\n"
      "64:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 65f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "cbnz x26, 66f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 66f\n"
      "65:"  // Height 3: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "66:"  // Height 3: input setup done
      "cmp x25, #0x10\n"
      "blt 71f\n"
      "ldr q0, [x24, #0x0]\n"
      "ldr q1, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q2, [x22, #0x0]\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "blt 69f\n"
      "67:"  // Height 3: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      "ldr q29, [x28, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      "ldr q28, [x28, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      "ldr q5, [x28, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      "ldr q4, [x28, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      "ldr q3, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      "ldr q31, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      "ldr q30, [x28, #0xd0]\n"
      ".inst 0x4fa0e3b3  // sdot v19.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4fa1e3b7  // sdot v23.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4fa2e3bb  // sdot v27.4s, v29.16b, v2.4b[1]\n"
      "ldr q29, [x28, #0xe0]\n"
      ".inst 0x4f80eb90  // sdot v16.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb94  // sdot v20.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f82eb98  // sdot v24.4s, v28.16b, v2.4b[2]\n"
      "ldr q28, [x28, #0xf0]\n"
      ".inst 0x4f80e8b1  // sdot v17.4s, v5.16b, v0.4b[2]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f81e8b5  // sdot v21.4s, v5.16b, v1.4b[2]\n"
      ".inst 0x4f82e8b9  // sdot v25.4s, v5.16b, v2.4b[2]\n"
      ".inst 0x4f80e892  // sdot v18.4s, v4.16b, v0.4b[2]\n"
      ".inst 0x4f81e896  // sdot v22.4s, v4.16b, v1.4b[2]\n"
      ".inst 0x4f82e89a  // sdot v26.4s, v4.16b, v2.4b[2]\n"
      ".inst 0x4f80e873  // sdot v19.4s, v3.16b, v0.4b[2]\n"
      ".inst 0x4f81e877  // sdot v23.4s, v3.16b, v1.4b[2]\n"
      ".inst 0x4f82e87b  // sdot v27.4s, v3.16b, v2.4b[2]\n"
      ".inst 0x4fa0ebf0  // sdot v16.4s, v31.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebf4  // sdot v20.4s, v31.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebf8  // sdot v24.4s, v31.16b, v2.4b[3]\n"
      ".inst 0x4fa0ebd1  // sdot v17.4s, v30.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebd5  // sdot v21.4s, v30.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebd9  // sdot v25.4s, v30.16b, v2.4b[3]\n"
      ".inst 0x4fa0ebb2  // sdot v18.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebb6  // sdot v22.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebba  // sdot v26.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4fa0eb93  // sdot v19.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb97  // sdot v23.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb9b  // sdot v27.4s, v28.16b, v2.4b[3]\n"
      "tbnz %x[flags], #31, 68f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "68:"  // Height 3: Multiply loop: unique 9: skip row sum
      "ldr q0, [x24, #0x0]\n"
      "ldr q1, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q2, [x22, #0x0]\n"
      "ldr q4, [x28, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "bge 67b\n"
      "69:"  // Height 3: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      "ldr q29, [x28, #0x70]\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      "ldr q28, [x28, #0x80]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      "ldr q5, [x28, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      "ldr q4, [x28, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      "ldr q3, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      "ldr q31, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      "ldr q30, [x28, #0xd0]\n"
      ".inst 0x4fa0e3b3  // sdot v19.4s, v29.16b, v0.4b[1]\n"
      ".inst 0x4fa1e3b7  // sdot v23.4s, v29.16b, v1.4b[1]\n"
      ".inst 0x4fa2e3bb  // sdot v27.4s, v29.16b, v2.4b[1]\n"
      "ldr q29, [x28, #0xe0]\n"
      ".inst 0x4f80eb90  // sdot v16.4s, v28.16b, v0.4b[2]\n"
      ".inst 0x4f81eb94  // sdot v20.4s, v28.16b, v1.4b[2]\n"
      ".inst 0x4f82eb98  // sdot v24.4s, v28.16b, v2.4b[2]\n"
      "ldr q28, [x28, #0xf0]\n"
      ".inst 0x4f80e8b1  // sdot v17.4s, v5.16b, v0.4b[2]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f81e8b5  // sdot v21.4s, v5.16b, v1.4b[2]\n"
      ".inst 0x4f82e8b9  // sdot v25.4s, v5.16b, v2.4b[2]\n"
      ".inst 0x4f80e892  // sdot v18.4s, v4.16b, v0.4b[2]\n"
      ".inst 0x4f81e896  // sdot v22.4s, v4.16b, v1.4b[2]\n"
      ".inst 0x4f82e89a  // sdot v26.4s, v4.16b, v2.4b[2]\n"
      ".inst 0x4f80e873  // sdot v19.4s, v3.16b, v0.4b[2]\n"
      ".inst 0x4f81e877  // sdot v23.4s, v3.16b, v1.4b[2]\n"
      ".inst 0x4f82e87b  // sdot v27.4s, v3.16b, v2.4b[2]\n"
      ".inst 0x4fa0ebf0  // sdot v16.4s, v31.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebf4  // sdot v20.4s, v31.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebf8  // sdot v24.4s, v31.16b, v2.4b[3]\n"
      ".inst 0x4fa0ebd1  // sdot v17.4s, v30.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebd5  // sdot v21.4s, v30.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebd9  // sdot v25.4s, v30.16b, v2.4b[3]\n"
      ".inst 0x4fa0ebb2  // sdot v18.4s, v29.16b, v0.4b[3]\n"
      ".inst 0x4fa1ebb6  // sdot v22.4s, v29.16b, v1.4b[3]\n"
      ".inst 0x4fa2ebba  // sdot v26.4s, v29.16b, v2.4b[3]\n"
      ".inst 0x4fa0eb93  // sdot v19.4s, v28.16b, v0.4b[3]\n"
      ".inst 0x4fa1eb97  // sdot v23.4s, v28.16b, v1.4b[3]\n"
      ".inst 0x4fa2eb9b  // sdot v27.4s, v28.16b, v2.4b[3]\n"
      "tbnz %x[flags], #31, 70f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "70:"  // Height 3: Multiply loop: unique 10: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "71:"  // Height 3: Multiply loop: Main loop skip
      "cbz x25, 78f\n"
      "cmp x25, #0x4\n"
      "blt 74f\n"
      "72:"  // Height 3: Multiply loop: Odd block loop
      "ldr s0, [x24], #0x4\n"
      "ldr s1, [x23], #0x4\n"
      "ldr s2, [x22], #0x4\n"
      "tbnz %x[flags], #31, 73f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "73:"  // Height 3: Multiply loop: unique 11: skip row sum
      "ldr q31, [x28, #0x0]\n"
      "ldr q30, [x28, #0x10]\n"
      "sub x25, x25, #0x4\n"
      "ldr q29, [x28, #0x20]\n"
      "ldr q28, [x28, #0x30]\n"
      "cmp x25, #0x4\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e3f0  // sdot v16.4s, v31.16b, v0.4b[0]\n"
      ".inst 0x4f81e3f4  // sdot v20.4s, v31.16b, v1.4b[0]\n"
      ".inst 0x4f82e3f8  // sdot v24.4s, v31.16b, v2.4b[0]\n"
      ".inst 0x4f80e3d1  // sdot v17.4s, v30.16b, v0.4b[0]\n"
      ".inst 0x4f81e3d5  // sdot v21.4s, v30.16b, v1.4b[0]\n"
      ".inst 0x4f82e3d9  // sdot v25.4s, v30.16b, v2.4b[0]\n"
      ".inst 0x4f80e3b2  // sdot v18.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f81e3b6  // sdot v22.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f82e3ba  // sdot v26.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f80e393  // sdot v19.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f81e397  // sdot v23.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f82e39b  // sdot v27.4s, v28.16b, v2.4b[0]\n"
      "bge 72b\n"
      "74:"  // Height 3: Multiply loop: Skip odd blocks
      "cbz x25, 78f\n"
      "tbz x25, #1, 75f\n"
      "ldr h0, [x24], #0x2\n"
      "ldr h1, [x23], #0x2\n"
      "ldr h2, [x22], #0x2\n"
      "tbz x25, #0, 76f\n"
      "ld1 { v0.b }[2], [x24]\n"
      "ld1 { v1.b }[2], [x23]\n"
      "ld1 { v2.b }[2], [x22]\n"
      "b 76f\n"
      "75:"  // Height 3: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x24, #0x0]\n"
      "ldr b1, [x23, #0x0]\n"
      "ldr b2, [x22, #0x0]\n"
      "76:"  // Height 3: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 77f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      "77:"  // Height 3: Multiply loop: unique 12: skip row sum
      "ldr q31, [x28, #0x0]\n"
      "ldr q30, [x28, #0x10]\n"
      "ldr q29, [x28, #0x20]\n"
      "ldr q28, [x28, #0x30]\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e3f0  // sdot v16.4s, v31.16b, v0.4b[0]\n"
      ".inst 0x4f81e3f4  // sdot v20.4s, v31.16b, v1.4b[0]\n"
      ".inst 0x4f82e3f8  // sdot v24.4s, v31.16b, v2.4b[0]\n"
      ".inst 0x4f80e3d1  // sdot v17.4s, v30.16b, v0.4b[0]\n"
      ".inst 0x4f81e3d5  // sdot v21.4s, v30.16b, v1.4b[0]\n"
      ".inst 0x4f82e3d9  // sdot v25.4s, v30.16b, v2.4b[0]\n"
      ".inst 0x4f80e3b2  // sdot v18.4s, v29.16b, v0.4b[0]\n"
      ".inst 0x4f81e3b6  // sdot v22.4s, v29.16b, v1.4b[0]\n"
      ".inst 0x4f82e3ba  // sdot v26.4s, v29.16b, v2.4b[0]\n"
      ".inst 0x4f80e393  // sdot v19.4s, v28.16b, v0.4b[0]\n"
      ".inst 0x4f81e397  // sdot v23.4s, v28.16b, v1.4b[0]\n"
      ".inst 0x4f82e39b  // sdot v27.4s, v28.16b, v2.4b[0]\n"
      "78:"  // Height 3: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 64b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "add x24, x27, x20\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x23, x24, x20\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "tbnz %x[flags], #31, 79f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "ld1r { v28.4s }, [x20]\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "neg v28.4s, v28.4s\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "mul v11.4s, v11.4s, v28.4s\n"
      "mul v12.4s, v12.4s, v28.4s\n"
      "mul v13.4s, v13.4s, v28.4s\n"
      "79:"  // Height 3: skip row sum fixup
      "ldr q0, [x10, #0x0]\n"
      "ldr q31, [x10, #0x10]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q30, [x10, #0x20]\n"
      "ldr q29, [x10, #0x30]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add v20.4s, v20.4s, v12.4s\n"
      "add v21.4s, v21.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v28.4s }, [x20]\n"
      "add v22.4s, v22.4s, v12.4s\n"
      "add v23.4s, v23.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add v24.4s, v24.4s, v13.4s\n"
      "add v25.4s, v25.4s, v13.4s\n"
      "add x10, x10, #0x40\n"
      "add v26.4s, v26.4s, v13.4s\n"
      "add v27.4s, v27.4s, v13.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v31.4s\n"
      "add v18.4s, v18.4s, v30.4s\n"
      "add v19.4s, v19.4s, v29.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v31.4s\n"
      "add v22.4s, v22.4s, v30.4s\n"
      "add v23.4s, v23.4s, v29.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v25.4s, v25.4s, v31.4s\n"
      "add v26.4s, v26.4s, v30.4s\n"
      "add v27.4s, v27.4s, v29.4s\n"
      "sqrdmulh v16.4s, v16.4s, v28.4s\n"
      "sqrdmulh v17.4s, v17.4s, v28.4s\n"
      "sqrdmulh v18.4s, v18.4s, v28.4s\n"
      "sqrdmulh v19.4s, v19.4s, v28.4s\n"
      "sqrdmulh v20.4s, v20.4s, v28.4s\n"
      "sqrdmulh v21.4s, v21.4s, v28.4s\n"
      "sqrdmulh v22.4s, v22.4s, v28.4s\n"
      "sqrdmulh v23.4s, v23.4s, v28.4s\n"
      "sqrdmulh v24.4s, v24.4s, v28.4s\n"
      "sqrdmulh v25.4s, v25.4s, v28.4s\n"
      "sqrdmulh v26.4s, v26.4s, v28.4s\n"
      "sqrdmulh v27.4s, v27.4s, v28.4s\n"
      "tbz %x[flags], #5, 80f\n"
      "and v1.16b, v16.16b, v0.16b\n"
      "and v31.16b, v17.16b, v0.16b\n"
      "and v30.16b, v18.16b, v0.16b\n"
      "and v29.16b, v19.16b, v0.16b\n"
      "and v28.16b, v20.16b, v0.16b\n"
      "and v3.16b, v21.16b, v0.16b\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sshr v31.4s, v31.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "and v2.16b, v22.16b, v0.16b\n"
      "sqadd v16.4s, v16.4s, v1.4s\n"
      "sqadd v17.4s, v17.4s, v31.4s\n"
      "sqadd v18.4s, v18.4s, v30.4s\n"
      "sqadd v19.4s, v19.4s, v29.4s\n"
      "sqadd v20.4s, v20.4s, v28.4s\n"
      "and v1.16b, v23.16b, v0.16b\n"
      "and v31.16b, v24.16b, v0.16b\n"
      "and v30.16b, v25.16b, v0.16b\n"
      "and v29.16b, v26.16b, v0.16b\n"
      "and v28.16b, v27.16b, v0.16b\n"
      "sshr v3.4s, v3.4s, #0x1f\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sshr v31.4s, v31.4s, #0x1f\n"
      "sshr v30.4s, v30.4s, #0x1f\n"
      "sshr v29.4s, v29.4s, #0x1f\n"
      "sshr v28.4s, v28.4s, #0x1f\n"
      "sqadd v21.4s, v21.4s, v3.4s\n"
      "sqadd v22.4s, v22.4s, v2.4s\n"
      "sqadd v23.4s, v23.4s, v1.4s\n"
      "sqadd v24.4s, v24.4s, v31.4s\n"
      "sqadd v25.4s, v25.4s, v30.4s\n"
      "sqadd v26.4s, v26.4s, v29.4s\n"
      "sqadd v27.4s, v27.4s, v28.4s\n"
      "80:"  // Height 3: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v30.4s }, [x21]\n"
      "ld1r { v29.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v28.4s }, [x20]\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "add v16.4s, v16.4s, v30.4s\n"
      "add v17.4s, v17.4s, v30.4s\n"
      "add v18.4s, v18.4s, v30.4s\n"
      "add v19.4s, v19.4s, v30.4s\n"
      "add v20.4s, v20.4s, v30.4s\n"
      "add v21.4s, v21.4s, v30.4s\n"
      "add v22.4s, v22.4s, v30.4s\n"
      "add v23.4s, v23.4s, v30.4s\n"
      "add v24.4s, v24.4s, v30.4s\n"
      "add v25.4s, v25.4s, v30.4s\n"
      "add v26.4s, v26.4s, v30.4s\n"
      "add v27.4s, v27.4s, v30.4s\n"
      "smin v16.4s, v16.4s, v29.4s\n"
      "smin v17.4s, v17.4s, v29.4s\n"
      "smin v18.4s, v18.4s, v29.4s\n"
      "smin v19.4s, v19.4s, v29.4s\n"
      "smin v20.4s, v20.4s, v29.4s\n"
      "smin v21.4s, v21.4s, v29.4s\n"
      "smin v22.4s, v22.4s, v29.4s\n"
      "smin v23.4s, v23.4s, v29.4s\n"
      "smin v24.4s, v24.4s, v29.4s\n"
      "smin v25.4s, v25.4s, v29.4s\n"
      "smin v26.4s, v26.4s, v29.4s\n"
      "smin v27.4s, v27.4s, v29.4s\n"
      "smax v16.4s, v16.4s, v28.4s\n"
      "smax v17.4s, v17.4s, v28.4s\n"
      "smax v18.4s, v18.4s, v28.4s\n"
      "smax v19.4s, v19.4s, v28.4s\n"
      "smax v20.4s, v20.4s, v28.4s\n"
      "smax v21.4s, v21.4s, v28.4s\n"
      "smax v22.4s, v22.4s, v28.4s\n"
      "smax v23.4s, v23.4s, v28.4s\n"
      "smax v24.4s, v24.4s, v28.4s\n"
      "smax v25.4s, v25.4s, v28.4s\n"
      "smax v26.4s, v26.4s, v28.4s\n"
      "smax v27.4s, v27.4s, v28.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v19.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v18.8h, v22.8h, v23.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v17.8h, v26.8h, v27.8h\n"
      "uzp1 v16.16b, v16.16b, v19.16b\n"
      "uzp1 v20.16b, v20.16b, v18.16b\n"
      "uzp1 v24.16b, v24.16b, v17.16b\n"
      "bge 89f\n"
      "tbz x9, #3, 84f\n"
      "str d16, [x27], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "tbz x9, #2, 82f\n"
      "st1 { v16.s }[2], [x27], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "tbz x9, #1, 81f\n"
      "st1 { v16.h }[6], [x27], #0x2\n"
      "st1 { v20.h }[6], [x24], #0x2\n"
      "st1 { v24.h }[6], [x23], #0x2\n"
      "tbz x9, #0, 88f\n"
      "st1 { v16.b }[14], [x27]\n"
      "st1 { v20.b }[14], [x24]\n"
      "st1 { v24.b }[14], [x23]\n"
      "b 88f\n"
      "81:"  // Height 3: Partial direct writeback: partial_1_12
      "tbz x9, #0, 88f\n"
      "st1 { v16.b }[12], [x27]\n"
      "st1 { v20.b }[12], [x24]\n"
      "st1 { v24.b }[12], [x23]\n"
      "b 88f\n"
      "82:"  // Height 3: Partial direct writeback: partial_2_8
      "tbz x9, #1, 83f\n"
      "st1 { v16.h }[4], [x27], #0x2\n"
      "st1 { v20.h }[4], [x24], #0x2\n"
      "st1 { v24.h }[4], [x23], #0x2\n"
      "tbz x9, #0, 88f\n"
      "st1 { v16.b }[10], [x27]\n"
      "st1 { v20.b }[10], [x24]\n"
      "st1 { v24.b }[10], [x23]\n"
      "b 88f\n"
      "83:"  // Height 3: Partial direct writeback: partial_1_8
      "tbz x9, #0, 88f\n"
      "st1 { v16.b }[8], [x27]\n"
      "st1 { v20.b }[8], [x24]\n"
      "st1 { v24.b }[8], [x23]\n"
      "b 88f\n"
      "84:"  // Height 3: Partial direct writeback: partial_4_0
      "tbz x9, #2, 86f\n"
      "str s16, [x27], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "tbz x9, #1, 85f\n"
      "st1 { v16.h }[2], [x27], #0x2\n"
      "st1 { v20.h }[2], [x24], #0x2\n"
      "st1 { v24.h }[2], [x23], #0x2\n"
      "tbz x9, #0, 88f\n"
      "st1 { v16.b }[6], [x27]\n"
      "st1 { v20.b }[6], [x24]\n"
      "st1 { v24.b }[6], [x23]\n"
      "b 88f\n"
      "85:"  // Height 3: Partial direct writeback: partial_1_4
      "tbz x9, #0, 88f\n"
      "st1 { v16.b }[4], [x27]\n"
      "st1 { v20.b }[4], [x24]\n"
      "st1 { v24.b }[4], [x23]\n"
      "b 88f\n"
      "86:"  // Height 3: Partial direct writeback: partial_2_0
      "tbz x9, #1, 87f\n"
      "str h16, [x27], #0x2\n"
      "str h20, [x24], #0x2\n"
      "str h24, [x23], #0x2\n"
      "tbz x9, #0, 88f\n"
      "st1 { v16.b }[2], [x27]\n"
      "st1 { v20.b }[2], [x24]\n"
      "st1 { v24.b }[2], [x23]\n"
      "b 88f\n"
      "87:"  // Height 3: Partial direct writeback: partial_1_0
      "str b16, [x27, #0x0]\n"
      "str b20, [x24, #0x0]\n"
      "str b24, [x23, #0x0]\n"
      "88:"  // Height 3: Partial direct writeback: Done
      "b 90f\n"
      "89:"  // Height 3: Full writeback
      "str q16, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q20, [x24, #0x0]\n"
      "str q24, [x23, #0x0]\n"
      "90:"  // Height 3: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 62b\n"
      "b 122f\n"
      "91:"  // Height 4
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x27, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x4\n"
      "mov x10, %x[col_bias]\n"
      "movi v11.4s, #0x0\n"
      "movi v12.4s, #0x0\n"
      "bic %x[flags], %x[flags], #0x80000000\n"
      "ldr x9, [%x[args_ptr], %[offsetof_N]]\n"
      "movi v13.4s, #0x0\n"
      "movi v14.4s, #0x0\n"
      "ldr x28, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x27\n"
      "movi v15.16b, #0x1\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "92:"  // Height 4: Column loop
      "movi v16.4s, #0x0\n"
      "movi v17.4s, #0x0\n"
      "movi v18.4s, #0x0\n"
      "movi v19.4s, #0x0\n"
      "movi v20.4s, #0x0\n"
      "movi v21.4s, #0x0\n"
      "movi v22.4s, #0x0\n"
      "movi v23.4s, #0x0\n"
      "movi v24.4s, #0x0\n"
      "movi v25.4s, #0x0\n"
      "movi v26.4s, #0x0\n"
      "movi v27.4s, #0x0\n"
      "movi v28.4s, #0x0\n"
      "movi v29.4s, #0x0\n"
      "movi v30.4s, #0x0\n"
      "movi v31.4s, #0x0\n"
      "93:"  // Height 4: setup done
      "mov x26, #0x0\n"
      "94:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w25, [x20, x26, LSL #0x2]\n"
      "tbz %x[flags], #3, 95f\n"
      "ldr x20, [%x[input_ptr], x26, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x24, [x20, #0x0]\n"
      "ldr x23, [x20, #0x8]\n"
      "ldr x22, [x20, #0x10]\n"
      "ldr x21, [x20, #0x18]\n"
      "cbnz x26, 96f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 96f\n"
      "95:"  // Height 4: setup direct input
      "mov x24, %x[input_ptr]\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "96:"  // Height 4: input setup done
      "cmp x25, #0x10\n"
      "blt 101f\n"
      "ldr q0, [x24, #0x0]\n"
      "ldr q1, [x23, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q2, [x22, #0x0]\n"
      "ldr q3, [x21, #0x0]\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "blt 99f\n"
      "97:"  // Height 4: Multiply loop: Main loop head
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "add x24, x24, #0x10\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      ".inst 0x4f83e09c  // sdot v28.4s, v4.16b, v3.4b[0]\n"
      "ldr q4, [x28, #0x70]\n"
      "add x22, x22, #0x10\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      ".inst 0x4f83e0bd  // sdot v29.4s, v5.16b, v3.4b[0]\n"
      "ldr q5, [x28, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0de  // sdot v30.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x28, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0ff  // sdot v31.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x28, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      ".inst 0x4fa3e11c  // sdot v28.4s, v8.16b, v3.4b[1]\n"
      "ldr q8, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      ".inst 0x4fa3e13d  // sdot v29.4s, v9.16b, v3.4b[1]\n"
      "ldr q9, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      ".inst 0x4fa3e15e  // sdot v30.4s, v10.16b, v3.4b[1]\n"
      "ldr q10, [x28, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      ".inst 0x4fa2e09b  // sdot v27.4s, v4.16b, v2.4b[1]\n"
      ".inst 0x4fa3e09f  // sdot v31.4s, v4.16b, v3.4b[1]\n"
      "ldr q4, [x28, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      ".inst 0x4f82e8b8  // sdot v24.4s, v5.16b, v2.4b[2]\n"
      ".inst 0x4f83e8bc  // sdot v28.4s, v5.16b, v3.4b[2]\n"
      "ldr q5, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d9  // sdot v25.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8dd  // sdot v29.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8fa  // sdot v26.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8fe  // sdot v30.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      ".inst 0x4f82e91b  // sdot v27.4s, v8.16b, v2.4b[2]\n"
      ".inst 0x4f83e91f  // sdot v31.4s, v8.16b, v3.4b[2]\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa2e938  // sdot v24.4s, v9.16b, v2.4b[3]\n"
      ".inst 0x4fa3e93c  // sdot v28.4s, v9.16b, v3.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa2e959  // sdot v25.4s, v10.16b, v2.4b[3]\n"
      ".inst 0x4fa3e95d  // sdot v29.4s, v10.16b, v3.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa2e89a  // sdot v26.4s, v4.16b, v2.4b[3]\n"
      ".inst 0x4fa3e89e  // sdot v30.4s, v4.16b, v3.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8bb  // sdot v27.4s, v5.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8bf  // sdot v31.4s, v5.16b, v3.4b[3]\n"
      "tbnz %x[flags], #31, 98f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "98:"  // Height 4: Multiply loop: unique 13: skip row sum
      "ldr q0, [x24, #0x0]\n"
      "ldr q1, [x23, #0x0]\n"
      "sub x25, x25, #0x10\n"
      "ldr q2, [x22, #0x0]\n"
      "ldr q3, [x21, #0x0]\n"
      "cmp x25, #0x20\n"
      "ldr q4, [x28, #0x0]\n"
      "ldr q5, [x28, #0x10]\n"
      "ldr q6, [x28, #0x20]\n"
      "ldr q7, [x28, #0x30]\n"
      "ldr q8, [x28, #0x40]\n"
      "ldr q9, [x28, #0x50]\n"
      "ldr q10, [x28, #0x60]\n"
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "bge 97b\n"
      "99:"  // Height 4: Multiply loop: Single iteration only
      ".inst 0x4f80e090  // sdot v16.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e094  // sdot v20.4s, v4.16b, v1.4b[0]\n"
      "sub x25, x25, #0x10\n"
      "add x24, x24, #0x10\n"
      ".inst 0x4f82e098  // sdot v24.4s, v4.16b, v2.4b[0]\n"
      ".inst 0x4f83e09c  // sdot v28.4s, v4.16b, v3.4b[0]\n"
      "ldr q4, [x28, #0x70]\n"
      "add x23, x23, #0x10\n"
      ".inst 0x4f80e0b1  // sdot v17.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b5  // sdot v21.4s, v5.16b, v1.4b[0]\n"
      "add x22, x22, #0x10\n"
      "add x21, x21, #0x10\n"
      ".inst 0x4f82e0b9  // sdot v25.4s, v5.16b, v2.4b[0]\n"
      ".inst 0x4f83e0bd  // sdot v29.4s, v5.16b, v3.4b[0]\n"
      "ldr q5, [x28, #0x80]\n"
      ".inst 0x4f80e0d2  // sdot v18.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d6  // sdot v22.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0da  // sdot v26.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0de  // sdot v30.4s, v6.16b, v3.4b[0]\n"
      "ldr q6, [x28, #0x90]\n"
      ".inst 0x4f80e0f3  // sdot v19.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f7  // sdot v23.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0fb  // sdot v27.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0ff  // sdot v31.4s, v7.16b, v3.4b[0]\n"
      "ldr q7, [x28, #0xa0]\n"
      ".inst 0x4fa0e110  // sdot v16.4s, v8.16b, v0.4b[1]\n"
      ".inst 0x4fa1e114  // sdot v20.4s, v8.16b, v1.4b[1]\n"
      ".inst 0x4fa2e118  // sdot v24.4s, v8.16b, v2.4b[1]\n"
      ".inst 0x4fa3e11c  // sdot v28.4s, v8.16b, v3.4b[1]\n"
      "ldr q8, [x28, #0xb0]\n"
      ".inst 0x4fa0e131  // sdot v17.4s, v9.16b, v0.4b[1]\n"
      ".inst 0x4fa1e135  // sdot v21.4s, v9.16b, v1.4b[1]\n"
      ".inst 0x4fa2e139  // sdot v25.4s, v9.16b, v2.4b[1]\n"
      ".inst 0x4fa3e13d  // sdot v29.4s, v9.16b, v3.4b[1]\n"
      "ldr q9, [x28, #0xc0]\n"
      ".inst 0x4fa0e152  // sdot v18.4s, v10.16b, v0.4b[1]\n"
      ".inst 0x4fa1e156  // sdot v22.4s, v10.16b, v1.4b[1]\n"
      ".inst 0x4fa2e15a  // sdot v26.4s, v10.16b, v2.4b[1]\n"
      ".inst 0x4fa3e15e  // sdot v30.4s, v10.16b, v3.4b[1]\n"
      "ldr q10, [x28, #0xd0]\n"
      ".inst 0x4fa0e093  // sdot v19.4s, v4.16b, v0.4b[1]\n"
      ".inst 0x4fa1e097  // sdot v23.4s, v4.16b, v1.4b[1]\n"
      ".inst 0x4fa2e09b  // sdot v27.4s, v4.16b, v2.4b[1]\n"
      ".inst 0x4fa3e09f  // sdot v31.4s, v4.16b, v3.4b[1]\n"
      "ldr q4, [x28, #0xe0]\n"
      ".inst 0x4f80e8b0  // sdot v16.4s, v5.16b, v0.4b[2]\n"
      ".inst 0x4f81e8b4  // sdot v20.4s, v5.16b, v1.4b[2]\n"
      ".inst 0x4f82e8b8  // sdot v24.4s, v5.16b, v2.4b[2]\n"
      ".inst 0x4f83e8bc  // sdot v28.4s, v5.16b, v3.4b[2]\n"
      "ldr q5, [x28, #0xf0]\n"
      "add x28, x28, #0x100\n"
      ".inst 0x4f80e8d1  // sdot v17.4s, v6.16b, v0.4b[2]\n"
      ".inst 0x4f81e8d5  // sdot v21.4s, v6.16b, v1.4b[2]\n"
      ".inst 0x4f82e8d9  // sdot v25.4s, v6.16b, v2.4b[2]\n"
      ".inst 0x4f83e8dd  // sdot v29.4s, v6.16b, v3.4b[2]\n"
      ".inst 0x4f80e8f2  // sdot v18.4s, v7.16b, v0.4b[2]\n"
      ".inst 0x4f81e8f6  // sdot v22.4s, v7.16b, v1.4b[2]\n"
      ".inst 0x4f82e8fa  // sdot v26.4s, v7.16b, v2.4b[2]\n"
      ".inst 0x4f83e8fe  // sdot v30.4s, v7.16b, v3.4b[2]\n"
      ".inst 0x4f80e913  // sdot v19.4s, v8.16b, v0.4b[2]\n"
      ".inst 0x4f81e917  // sdot v23.4s, v8.16b, v1.4b[2]\n"
      ".inst 0x4f82e91b  // sdot v27.4s, v8.16b, v2.4b[2]\n"
      ".inst 0x4f83e91f  // sdot v31.4s, v8.16b, v3.4b[2]\n"
      ".inst 0x4fa0e930  // sdot v16.4s, v9.16b, v0.4b[3]\n"
      ".inst 0x4fa1e934  // sdot v20.4s, v9.16b, v1.4b[3]\n"
      ".inst 0x4fa2e938  // sdot v24.4s, v9.16b, v2.4b[3]\n"
      ".inst 0x4fa3e93c  // sdot v28.4s, v9.16b, v3.4b[3]\n"
      ".inst 0x4fa0e951  // sdot v17.4s, v10.16b, v0.4b[3]\n"
      ".inst 0x4fa1e955  // sdot v21.4s, v10.16b, v1.4b[3]\n"
      ".inst 0x4fa2e959  // sdot v25.4s, v10.16b, v2.4b[3]\n"
      ".inst 0x4fa3e95d  // sdot v29.4s, v10.16b, v3.4b[3]\n"
      ".inst 0x4fa0e892  // sdot v18.4s, v4.16b, v0.4b[3]\n"
      ".inst 0x4fa1e896  // sdot v22.4s, v4.16b, v1.4b[3]\n"
      ".inst 0x4fa2e89a  // sdot v26.4s, v4.16b, v2.4b[3]\n"
      ".inst 0x4fa3e89e  // sdot v30.4s, v4.16b, v3.4b[3]\n"
      ".inst 0x4fa0e8b3  // sdot v19.4s, v5.16b, v0.4b[3]\n"
      ".inst 0x4fa1e8b7  // sdot v23.4s, v5.16b, v1.4b[3]\n"
      ".inst 0x4fa2e8bb  // sdot v27.4s, v5.16b, v2.4b[3]\n"
      ".inst 0x4fa3e8bf  // sdot v31.4s, v5.16b, v3.4b[3]\n"
      "tbnz %x[flags], #31, 100f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "100:"  // Height 4: Multiply loop: unique 14: skip row sum
      "prfm pldl1keep, [x24, #0x80]\n"
      "prfm pldl1keep, [x23, #0x80]\n"
      "prfm pldl1keep, [x22, #0x80]\n"
      "prfm pldl1keep, [x21, #0x80]\n"
      "101:"  // Height 4: Multiply loop: Main loop skip
      "cbz x25, 108f\n"
      "cmp x25, #0x4\n"
      "blt 104f\n"
      "102:"  // Height 4: Multiply loop: Odd block loop
      "ldr s0, [x24], #0x4\n"
      "ldr s1, [x23], #0x4\n"
      "ldr s2, [x22], #0x4\n"
      "ldr s3, [x21], #0x4\n"
      "tbnz %x[flags], #31, 103f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "103:"  // Height 4: Multiply loop: unique 15: skip row sum
      "ldr q7, [x28, #0x0]\n"
      "ldr q6, [x28, #0x10]\n"
      "sub x25, x25, #0x4\n"
      "ldr q5, [x28, #0x20]\n"
      "ldr q4, [x28, #0x30]\n"
      "cmp x25, #0x4\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e0f0  // sdot v16.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f4  // sdot v20.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f8  // sdot v24.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0fc  // sdot v28.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f80e0d1  // sdot v17.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d5  // sdot v21.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d9  // sdot v25.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0dd  // sdot v29.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f80e0b2  // sdot v18.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b6  // sdot v22.4s, v5.16b, v1.4b[0]\n"
      ".inst 0x4f82e0ba  // sdot v26.4s, v5.16b, v2.4b[0]\n"
      ".inst 0x4f83e0be  // sdot v30.4s, v5.16b, v3.4b[0]\n"
      ".inst 0x4f80e093  // sdot v19.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e097  // sdot v23.4s, v4.16b, v1.4b[0]\n"
      ".inst 0x4f82e09b  // sdot v27.4s, v4.16b, v2.4b[0]\n"
      ".inst 0x4f83e09f  // sdot v31.4s, v4.16b, v3.4b[0]\n"
      "bge 102b\n"
      "104:"  // Height 4: Multiply loop: Skip odd blocks
      "cbz x25, 108f\n"
      "tbz x25, #1, 105f\n"
      "ldr h0, [x24], #0x2\n"
      "ldr h1, [x23], #0x2\n"
      "ldr h2, [x22], #0x2\n"
      "ldr h3, [x21], #0x2\n"
      "tbz x25, #0, 106f\n"
      "ld1 { v0.b }[2], [x24]\n"
      "ld1 { v1.b }[2], [x23]\n"
      "ld1 { v2.b }[2], [x22]\n"
      "ld1 { v3.b }[2], [x21]\n"
      "b 106f\n"
      "105:"  // Height 4: Multiply loop: Ragged operand read: partial_1_0
      "ldr b0, [x24, #0x0]\n"
      "ldr b1, [x23, #0x0]\n"
      "ldr b2, [x22, #0x0]\n"
      "ldr b3, [x21, #0x0]\n"
      "106:"  // Height 4: Multiply loop: Ragged operand read: Done
      "tbnz %x[flags], #31, 107f\n"
      ".inst 0x4e8f940b  // sdot v11.4s, v0.16b, v15.16b\n"
      ".inst 0x4e8f942c  // sdot v12.4s, v1.16b, v15.16b\n"
      ".inst 0x4e8f944d  // sdot v13.4s, v2.16b, v15.16b\n"
      ".inst 0x4e8f946e  // sdot v14.4s, v3.16b, v15.16b\n"
      "107:"  // Height 4: Multiply loop: unique 16: skip row sum
      "ldr q7, [x28, #0x0]\n"
      "ldr q6, [x28, #0x10]\n"
      "ldr q5, [x28, #0x20]\n"
      "ldr q4, [x28, #0x30]\n"
      "add x28, x28, #0x40\n"
      ".inst 0x4f80e0f0  // sdot v16.4s, v7.16b, v0.4b[0]\n"
      ".inst 0x4f81e0f4  // sdot v20.4s, v7.16b, v1.4b[0]\n"
      ".inst 0x4f82e0f8  // sdot v24.4s, v7.16b, v2.4b[0]\n"
      ".inst 0x4f83e0fc  // sdot v28.4s, v7.16b, v3.4b[0]\n"
      ".inst 0x4f80e0d1  // sdot v17.4s, v6.16b, v0.4b[0]\n"
      ".inst 0x4f81e0d5  // sdot v21.4s, v6.16b, v1.4b[0]\n"
      ".inst 0x4f82e0d9  // sdot v25.4s, v6.16b, v2.4b[0]\n"
      ".inst 0x4f83e0dd  // sdot v29.4s, v6.16b, v3.4b[0]\n"
      ".inst 0x4f80e0b2  // sdot v18.4s, v5.16b, v0.4b[0]\n"
      ".inst 0x4f81e0b6  // sdot v22.4s, v5.16b, v1.4b[0]\n"
      ".inst 0x4f82e0ba  // sdot v26.4s, v5.16b, v2.4b[0]\n"
      ".inst 0x4f83e0be  // sdot v30.4s, v5.16b, v3.4b[0]\n"
      ".inst 0x4f80e093  // sdot v19.4s, v4.16b, v0.4b[0]\n"
      ".inst 0x4f81e097  // sdot v23.4s, v4.16b, v1.4b[0]\n"
      ".inst 0x4f82e09b  // sdot v27.4s, v4.16b, v2.4b[0]\n"
      ".inst 0x4f83e09f  // sdot v31.4s, v4.16b, v3.4b[0]\n"
      "108:"  // Height 4: Multiply loop: No odd multiplies
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x26, x26, #0x1\n"
      "cmp x26, x20\n"
      "bne 94b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "prfm pstl1keep, [x27, #0x0]\n"
      "add x24, x27, x20\n"
      "prfm pstl1keep, [x24, #0x0]\n"
      "add x23, x24, x20\n"
      "prfm pstl1keep, [x23, #0x0]\n"
      "add x22, x23, x20\n"
      "prfm pstl1keep, [x22, #0x0]\n"
      "tbnz %x[flags], #31, 109f\n"
      "add x20, %x[qp], %[b_offset]\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "addp v14.4s, v14.4s, v14.4s\n"
      "neg v0.4s, v0.4s\n"
      "addp v11.4s, v11.4s, v11.4s\n"
      "addp v12.4s, v12.4s, v12.4s\n"
      "addp v13.4s, v13.4s, v13.4s\n"
      "addp v14.4s, v14.4s, v14.4s\n"
      "mul v11.4s, v11.4s, v0.4s\n"
      "mul v12.4s, v12.4s, v0.4s\n"
      "mul v13.4s, v13.4s, v0.4s\n"
      "mul v14.4s, v14.4s, v0.4s\n"
      "109:"  // Height 4: skip row sum fixup
      "ldr q0, [x10, #0x0]\n"
      "ldr q4, [x10, #0x10]\n"
      "add v16.4s, v16.4s, v11.4s\n"
      "add v17.4s, v17.4s, v11.4s\n"
      "ldr q3, [x10, #0x20]\n"
      "ldr q2, [x10, #0x30]\n"
      "add v18.4s, v18.4s, v11.4s\n"
      "add v19.4s, v19.4s, v11.4s\n"
      "add v20.4s, v20.4s, v12.4s\n"
      "add v21.4s, v21.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "orr %x[flags], %x[flags], #0x80000000\n"
      "ld1r { v1.4s }, [x20]\n"
      "add v22.4s, v22.4s, v12.4s\n"
      "add v23.4s, v23.4s, v12.4s\n"
      "add x20, %x[qp], %[per_layer_right_shift]\n"
      "add v24.4s, v24.4s, v13.4s\n"
      "add v25.4s, v25.4s, v13.4s\n"
      "add x10, x10, #0x40\n"
      "add v26.4s, v26.4s, v13.4s\n"
      "add v27.4s, v27.4s, v13.4s\n"
      "add v28.4s, v28.4s, v14.4s\n"
      "add v29.4s, v29.4s, v14.4s\n"
      "add v30.4s, v30.4s, v14.4s\n"
      "add v31.4s, v31.4s, v14.4s\n"
      "add v16.4s, v16.4s, v0.4s\n"
      "add v17.4s, v17.4s, v4.4s\n"
      "add v18.4s, v18.4s, v3.4s\n"
      "add v19.4s, v19.4s, v2.4s\n"
      "add v20.4s, v20.4s, v0.4s\n"
      "add v21.4s, v21.4s, v4.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v23.4s, v23.4s, v2.4s\n"
      "add v24.4s, v24.4s, v0.4s\n"
      "add v25.4s, v25.4s, v4.4s\n"
      "add v26.4s, v26.4s, v3.4s\n"
      "add v27.4s, v27.4s, v2.4s\n"
      "add v28.4s, v28.4s, v0.4s\n"
      "ld1r { v0.4s }, [x20]\n"
      "add v29.4s, v29.4s, v4.4s\n"
      "add v30.4s, v30.4s, v3.4s\n"
      "add v31.4s, v31.4s, v2.4s\n"
      "sqrdmulh v16.4s, v16.4s, v1.4s\n"
      "sqrdmulh v17.4s, v17.4s, v1.4s\n"
      "sqrdmulh v18.4s, v18.4s, v1.4s\n"
      "sqrdmulh v19.4s, v19.4s, v1.4s\n"
      "sqrdmulh v20.4s, v20.4s, v1.4s\n"
      "sqrdmulh v21.4s, v21.4s, v1.4s\n"
      "sqrdmulh v22.4s, v22.4s, v1.4s\n"
      "sqrdmulh v23.4s, v23.4s, v1.4s\n"
      "sqrdmulh v24.4s, v24.4s, v1.4s\n"
      "sqrdmulh v25.4s, v25.4s, v1.4s\n"
      "sqrdmulh v26.4s, v26.4s, v1.4s\n"
      "sqrdmulh v27.4s, v27.4s, v1.4s\n"
      "sqrdmulh v28.4s, v28.4s, v1.4s\n"
      "sqrdmulh v29.4s, v29.4s, v1.4s\n"
      "sqrdmulh v30.4s, v30.4s, v1.4s\n"
      "sqrdmulh v31.4s, v31.4s, v1.4s\n"
      "tbz %x[flags], #5, 110f\n"
      "and v2.16b, v16.16b, v0.16b\n"
      "and v1.16b, v17.16b, v0.16b\n"
      "and v7.16b, v18.16b, v0.16b\n"
      "and v6.16b, v19.16b, v0.16b\n"
      "and v5.16b, v20.16b, v0.16b\n"
      "and v4.16b, v21.16b, v0.16b\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "and v3.16b, v22.16b, v0.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sqadd v16.4s, v16.4s, v2.4s\n"
      "sqadd v17.4s, v17.4s, v1.4s\n"
      "and v2.16b, v23.16b, v0.16b\n"
      "and v1.16b, v24.16b, v0.16b\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sshr v3.4s, v3.4s, #0x1f\n"
      "sqadd v18.4s, v18.4s, v7.4s\n"
      "sqadd v19.4s, v19.4s, v6.4s\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sqadd v20.4s, v20.4s, v5.4s\n"
      "sqadd v21.4s, v21.4s, v4.4s\n"
      "sqadd v22.4s, v22.4s, v3.4s\n"
      "and v7.16b, v25.16b, v0.16b\n"
      "sqadd v23.4s, v23.4s, v2.4s\n"
      "sqadd v24.4s, v24.4s, v1.4s\n"
      "and v6.16b, v26.16b, v0.16b\n"
      "and v5.16b, v27.16b, v0.16b\n"
      "and v4.16b, v28.16b, v0.16b\n"
      "and v3.16b, v29.16b, v0.16b\n"
      "and v2.16b, v30.16b, v0.16b\n"
      "and v1.16b, v31.16b, v0.16b\n"
      "sshr v7.4s, v7.4s, #0x1f\n"
      "sshr v6.4s, v6.4s, #0x1f\n"
      "sshr v5.4s, v5.4s, #0x1f\n"
      "sshr v4.4s, v4.4s, #0x1f\n"
      "sshr v3.4s, v3.4s, #0x1f\n"
      "sshr v2.4s, v2.4s, #0x1f\n"
      "sshr v1.4s, v1.4s, #0x1f\n"
      "sqadd v25.4s, v25.4s, v7.4s\n"
      "sqadd v26.4s, v26.4s, v6.4s\n"
      "sqadd v27.4s, v27.4s, v5.4s\n"
      "sqadd v28.4s, v28.4s, v4.4s\n"
      "sqadd v29.4s, v29.4s, v3.4s\n"
      "sqadd v30.4s, v30.4s, v2.4s\n"
      "sqadd v31.4s, v31.4s, v1.4s\n"
      "110:"  // Height 4: no shift correction
      "add x21, %x[qp], %[c_offset]\n"
      "srshl v16.4s, v16.4s, v0.4s\n"
      "srshl v17.4s, v17.4s, v0.4s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1r { v3.4s }, [x21]\n"
      "ld1r { v2.4s }, [x20]\n"
      "srshl v18.4s, v18.4s, v0.4s\n"
      "srshl v19.4s, v19.4s, v0.4s\n"
      "srshl v20.4s, v20.4s, v0.4s\n"
      "srshl v21.4s, v21.4s, v0.4s\n"
      "add x20, %x[qp], %[minval]\n"
      "cmp x9, #0x10\n"
      "ld1r { v1.4s }, [x20]\n"
      "srshl v22.4s, v22.4s, v0.4s\n"
      "srshl v23.4s, v23.4s, v0.4s\n"
      "srshl v24.4s, v24.4s, v0.4s\n"
      "srshl v25.4s, v25.4s, v0.4s\n"
      "srshl v26.4s, v26.4s, v0.4s\n"
      "srshl v27.4s, v27.4s, v0.4s\n"
      "srshl v28.4s, v28.4s, v0.4s\n"
      "srshl v29.4s, v29.4s, v0.4s\n"
      "srshl v30.4s, v30.4s, v0.4s\n"
      "srshl v31.4s, v31.4s, v0.4s\n"
      "add v16.4s, v16.4s, v3.4s\n"
      "add v17.4s, v17.4s, v3.4s\n"
      "add v18.4s, v18.4s, v3.4s\n"
      "add v19.4s, v19.4s, v3.4s\n"
      "add v20.4s, v20.4s, v3.4s\n"
      "add v21.4s, v21.4s, v3.4s\n"
      "add v22.4s, v22.4s, v3.4s\n"
      "add v23.4s, v23.4s, v3.4s\n"
      "add v24.4s, v24.4s, v3.4s\n"
      "add v25.4s, v25.4s, v3.4s\n"
      "add v26.4s, v26.4s, v3.4s\n"
      "add v27.4s, v27.4s, v3.4s\n"
      "add v28.4s, v28.4s, v3.4s\n"
      "add v29.4s, v29.4s, v3.4s\n"
      "add v30.4s, v30.4s, v3.4s\n"
      "add v31.4s, v31.4s, v3.4s\n"
      "smin v16.4s, v16.4s, v2.4s\n"
      "smin v17.4s, v17.4s, v2.4s\n"
      "smin v18.4s, v18.4s, v2.4s\n"
      "smin v19.4s, v19.4s, v2.4s\n"
      "smin v20.4s, v20.4s, v2.4s\n"
      "smin v21.4s, v21.4s, v2.4s\n"
      "smin v22.4s, v22.4s, v2.4s\n"
      "smin v23.4s, v23.4s, v2.4s\n"
      "smin v24.4s, v24.4s, v2.4s\n"
      "smin v25.4s, v25.4s, v2.4s\n"
      "smin v26.4s, v26.4s, v2.4s\n"
      "smin v27.4s, v27.4s, v2.4s\n"
      "smin v28.4s, v28.4s, v2.4s\n"
      "smin v29.4s, v29.4s, v2.4s\n"
      "smin v30.4s, v30.4s, v2.4s\n"
      "smin v31.4s, v31.4s, v2.4s\n"
      "smax v16.4s, v16.4s, v1.4s\n"
      "smax v17.4s, v17.4s, v1.4s\n"
      "smax v18.4s, v18.4s, v1.4s\n"
      "smax v19.4s, v19.4s, v1.4s\n"
      "smax v20.4s, v20.4s, v1.4s\n"
      "smax v21.4s, v21.4s, v1.4s\n"
      "smax v22.4s, v22.4s, v1.4s\n"
      "smax v23.4s, v23.4s, v1.4s\n"
      "smax v24.4s, v24.4s, v1.4s\n"
      "smax v25.4s, v25.4s, v1.4s\n"
      "smax v26.4s, v26.4s, v1.4s\n"
      "smax v27.4s, v27.4s, v1.4s\n"
      "smax v28.4s, v28.4s, v1.4s\n"
      "smax v29.4s, v29.4s, v1.4s\n"
      "smax v30.4s, v30.4s, v1.4s\n"
      "smax v31.4s, v31.4s, v1.4s\n"
      "uzp1 v16.8h, v16.8h, v17.8h\n"
      "uzp1 v0.8h, v18.8h, v19.8h\n"
      "uzp1 v20.8h, v20.8h, v21.8h\n"
      "uzp1 v19.8h, v22.8h, v23.8h\n"
      "uzp1 v24.8h, v24.8h, v25.8h\n"
      "uzp1 v18.8h, v26.8h, v27.8h\n"
      "uzp1 v28.8h, v28.8h, v29.8h\n"
      "uzp1 v17.8h, v30.8h, v31.8h\n"
      "uzp1 v16.16b, v16.16b, v0.16b\n"
      "uzp1 v20.16b, v20.16b, v19.16b\n"
      "uzp1 v24.16b, v24.16b, v18.16b\n"
      "uzp1 v28.16b, v28.16b, v17.16b\n"
      "bge 119f\n"
      "tbz x9, #3, 114f\n"
      "str d16, [x27], #0x8\n"
      "str d20, [x24], #0x8\n"
      "str d24, [x23], #0x8\n"
      "str d28, [x22], #0x8\n"
      "tbz x9, #2, 112f\n"
      "st1 { v16.s }[2], [x27], #0x4\n"
      "st1 { v20.s }[2], [x24], #0x4\n"
      "st1 { v24.s }[2], [x23], #0x4\n"
      "st1 { v28.s }[2], [x22], #0x4\n"
      "tbz x9, #1, 111f\n"
      "st1 { v16.h }[6], [x27], #0x2\n"
      "st1 { v20.h }[6], [x24], #0x2\n"
      "st1 { v24.h }[6], [x23], #0x2\n"
      "st1 { v28.h }[6], [x22], #0x2\n"
      "tbz x9, #0, 118f\n"
      "st1 { v16.b }[14], [x27]\n"
      "st1 { v20.b }[14], [x24]\n"
      "st1 { v24.b }[14], [x23]\n"
      "st1 { v28.b }[14], [x22]\n"
      "b 118f\n"
      "111:"  // Height 4: Partial direct writeback: partial_1_12
      "tbz x9, #0, 118f\n"
      "st1 { v16.b }[12], [x27]\n"
      "st1 { v20.b }[12], [x24]\n"
      "st1 { v24.b }[12], [x23]\n"
      "st1 { v28.b }[12], [x22]\n"
      "b 118f\n"
      "112:"  // Height 4: Partial direct writeback: partial_2_8
      "tbz x9, #1, 113f\n"
      "st1 { v16.h }[4], [x27], #0x2\n"
      "st1 { v20.h }[4], [x24], #0x2\n"
      "st1 { v24.h }[4], [x23], #0x2\n"
      "st1 { v28.h }[4], [x22], #0x2\n"
      "tbz x9, #0, 118f\n"
      "st1 { v16.b }[10], [x27]\n"
      "st1 { v20.b }[10], [x24]\n"
      "st1 { v24.b }[10], [x23]\n"
      "st1 { v28.b }[10], [x22]\n"
      "b 118f\n"
      "113:"  // Height 4: Partial direct writeback: partial_1_8
      "tbz x9, #0, 118f\n"
      "st1 { v16.b }[8], [x27]\n"
      "st1 { v20.b }[8], [x24]\n"
      "st1 { v24.b }[8], [x23]\n"
      "st1 { v28.b }[8], [x22]\n"
      "b 118f\n"
      "114:"  // Height 4: Partial direct writeback: partial_4_0
      "tbz x9, #2, 116f\n"
      "str s16, [x27], #0x4\n"
      "str s20, [x24], #0x4\n"
      "str s24, [x23], #0x4\n"
      "str s28, [x22], #0x4\n"
      "tbz x9, #1, 115f\n"
      "st1 { v16.h }[2], [x27], #0x2\n"
      "st1 { v20.h }[2], [x24], #0x2\n"
      "st1 { v24.h }[2], [x23], #0x2\n"
      "st1 { v28.h }[2], [x22], #0x2\n"
      "tbz x9, #0, 118f\n"
      "st1 { v16.b }[6], [x27]\n"
      "st1 { v20.b }[6], [x24]\n"
      "st1 { v24.b }[6], [x23]\n"
      "st1 { v28.b }[6], [x22]\n"
      "b 118f\n"
      "115:"  // Height 4: Partial direct writeback: partial_1_4
      "tbz x9, #0, 118f\n"
      "st1 { v16.b }[4], [x27]\n"
      "st1 { v20.b }[4], [x24]\n"
      "st1 { v24.b }[4], [x23]\n"
      "st1 { v28.b }[4], [x22]\n"
      "b 118f\n"
      "116:"  // Height 4: Partial direct writeback: partial_2_0
      "tbz x9, #1, 117f\n"
      "str h16, [x27], #0x2\n"
      "str h20, [x24], #0x2\n"
      "str h24, [x23], #0x2\n"
      "str h28, [x22], #0x2\n"
      "tbz x9, #0, 118f\n"
      "st1 { v16.b }[2], [x27]\n"
      "st1 { v20.b }[2], [x24]\n"
      "st1 { v24.b }[2], [x23]\n"
      "st1 { v28.b }[2], [x22]\n"
      "b 118f\n"
      "117:"  // Height 4: Partial direct writeback: partial_1_0
      "str b16, [x27, #0x0]\n"
      "str b20, [x24, #0x0]\n"
      "str b24, [x23, #0x0]\n"
      "str b28, [x22, #0x0]\n"
      "118:"  // Height 4: Partial direct writeback: Done
      "b 120f\n"
      "119:"  // Height 4: Full writeback
      "str q16, [x27, #0x0]\n"
      "add x27, x27, #0x10\n"
      "str q20, [x24, #0x0]\n"
      "str q24, [x23, #0x0]\n"
      "str q28, [x22, #0x0]\n"
      "120:"  // Height 4: Writeback done
      "subs x9, x9, #0x10\n"
      "bgt 92b\n"
      "subs %x[M], %x[M], #0x4\n"
      "beq 122f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 121f\n"
      "add x21, x21, #0x4\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "121:"  // Update direct input
      "mov x20, #0x4\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "122:"  // Exit
      : [M] "+&r" (M), [flags] "+&r" (flags), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [b_offset] "I" (offsetof(Requantize32, b_offset)), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "v0", "v1", "v2", "v3", "v4", "v5", "v6", "v7", "v8", "v9", "v10", "v11", "v12", "v13", "v14", "v15", "v16", "v17", "v18", "v19", "v20", "v21", "v22", "v23", "v24", "v25", "v26", "v27", "v28", "v29", "v30", "v31", "x9", "x10", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28"
    );
}

} // namespace arm_gemm
#endif // __aarch64__
