#!/usr/bin/env python

"""
Handlers for a resource.

Copyright (C) 2014, 2015, 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from email.mime.text import MIMEText
from imiptools.data import get_address, uri_dict
from imiptools.handlers import Handler
from imiptools.handlers.common import CommonFreebusy, CommonEvent
from imiptools.handlers.scheduling import apply_scheduling_functions, \
                                          confirm_scheduling, \
                                          finish_scheduling, \
                                          retract_scheduling

class ResourceHandler(CommonEvent, Handler):

    "Handling mechanisms specific to resources."

    def _process(self, handle_for_attendee):

        """
        Record details from the incoming message, using the given
        'handle_for_attendee' callable to process any valid message
        appropriately.
        """

        oa = self.require_organiser_and_attendees()
        if not oa:
            return None

        organiser_item, attendees = oa

        # Process for the current user, a resource as attendee.

        if not self.have_new_object():
            return None

        # Collect response objects produced when handling the request.

        handle_for_attendee()

    def _add_for_attendee(self):

        """
        Attempt to add a recurrence to an existing object for the current user.
        This does not request a response concerning participation, apparently.
        """

        # Request details where configured, doing so for unknown objects anyway.

        if self.will_refresh():
            self.make_refresh()
            return

        # Record the event as a recurrence of the parent object.

        self.update_recurrenceid()
        self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())

        # Remove any previous cancellations involving this event.

        self.store.remove_cancellation(self.user, self.uid, self.recurrenceid)

        # Update free/busy information.

        self.update_event_in_freebusy(for_organiser=False)

        # Confirm the scheduling of the recurrence.

        self.confirm_scheduling()

    def _schedule_for_attendee(self):

        "Attempt to schedule the current object for the current user."

        attendee_attr = uri_dict(self.obj.get_value_map("ATTENDEE"))[self.user]
        delegates = None

        # Attempt to schedule the event.

        try:
            scheduled, description = self.schedule()

            # Update the participation of the resource in the object.
            # Update free/busy information.

            if scheduled in ("ACCEPTED", "DECLINED"):
                method = "REPLY"
                attendee_attr = self.update_participation(scheduled)

                self.update_event_in_freebusy(for_organiser=False)
                self.remove_event_from_freebusy_offers()

                # Set the complete event or an additional occurrence.

                event = self.obj.to_node()
                self.store.set_event(self.user, self.uid, self.recurrenceid, event)

                # Remove additional recurrences if handling a complete event.
                # Also remove any previous cancellations involving this event.

                if not self.recurrenceid:
                    self.store.remove_recurrences(self.user, self.uid)
                    self.store.remove_cancellations(self.user, self.uid)
                else:
                    self.store.remove_cancellation(self.user, self.uid, self.recurrenceid)

                if scheduled == "ACCEPTED":
                    self.confirm_scheduling()

            # For delegated proposals, prepare a request to the delegates in
            # addition to the usual response.

            elif scheduled == "DELEGATED":
                method = "REPLY"
                attendee_attr = self.update_participation("DELEGATED")

                # The recipient will have indicated the delegate whose details
                # will have been added to the object.

                delegates = attendee_attr["DELEGATED-TO"]

            # For countered proposals, record the offer in the resource's
            # free/busy collection.

            elif scheduled == "COUNTER":
                method = "COUNTER"
                self.update_event_in_freebusy_offers()

            # For inappropriate periods, reply declining participation.

            else:
                method = "REPLY"
                attendee_attr = self.update_participation("DECLINED")

        # Confirm any scheduling.

        finally:
            self.finish_scheduling()

        # Determine the recipients of the outgoing messages.

        recipients = map(get_address, self.obj.get_values("ORGANIZER"))

        # Add any description of the scheduling decision.

        self.add_result(None, recipients, MIMEText(description))

        # Make a version of the object with just this attendee, update the
        # DTSTAMP in the response, and return the object for sending.

        self.update_sender(attendee_attr)
        attendees = [(self.user, attendee_attr)]

        # Add delegates if delegating (RFC 5546 being inconsistent here since
        # it provides an example reply to the organiser without the delegate).

        if delegates:
            for delegate in delegates:
                delegate_attr = uri_dict(self.obj.get_value_map("ATTENDEE"))[delegate]
                attendees.append((delegate, delegate_attr))

        # Reply to the delegator in addition to the organiser if replying to a
        # delegation request.

        delegators = self.is_delegation()
        if delegators:
            for delegator in delegators:
                delegator_attr = uri_dict(self.obj.get_value_map("ATTENDEE"))[delegator]
                attendees.append((delegator, delegator_attr))
                recipients.append(get_address(delegator))

        # Prepare the response for the organiser plus any delegator.

        self.obj["ATTENDEE"] = attendees
        self.update_dtstamp()
        self.add_result(method, recipients, self.object_to_part(method, self.obj))

        # If delegating, send a request to the delegates.

        if delegates:
            method = "REQUEST"
            self.add_result(method, map(get_address, delegates), self.object_to_part(method, self.obj))

    def _cancel_for_attendee(self):

        """
        Cancel for the current user their attendance of the event described by
        the current object.
        """

        # Update free/busy information.

        self.remove_event_from_freebusy()

        # Update the stored event and cancel it.

        self.store.set_event(self.user, self.uid, self.recurrenceid, self.obj.to_node())
        self.store.cancel_event(self.user, self.uid, self.recurrenceid)

        # Retract the scheduling of the event.

        self.retract_scheduling()

    def _revoke_for_attendee(self):

        "Revoke any counter-proposal recorded as a free/busy offer."

        self.remove_event_from_freebusy_offers()

    # Scheduling details.

    def get_scheduling_functions(self):

        "Return the scheduling functions for the resource."

        return self.get_preferences().get("scheduling_function",
            "schedule_in_freebusy").split("\n")

    def schedule(self):

        """
        Attempt to schedule the current object, returning an indication of the
        kind of response to be returned: "COUNTER" for counter-proposals,
        "ACCEPTED" for acceptances, "DECLINED" for rejections, and None for
        invalid requests.
        """

        return apply_scheduling_functions(self)

    def confirm_scheduling(self):

        "Confirm that this event has been scheduled."

        confirm_scheduling(self)

    def finish_scheduling(self):

        "Finish the scheduling, unlocking resources where appropriate."

        finish_scheduling(self)

    def retract_scheduling(self):

        "Retract this event from scheduling records."

        retract_scheduling(self)

class Event(ResourceHandler):

    "An event handler."

    def add(self):

        "Add a new occurrence to an existing event."

        self._process(self._add_for_attendee)

    def cancel(self):

        "Cancel attendance for attendees."

        self._process(self._cancel_for_attendee)

    def counter(self):

        "Since this handler does not send requests, it will not handle replies."

        pass

    def declinecounter(self):

        "Revoke any counter-proposal."

        self._process(self._revoke_for_attendee)

    def publish(self):

        """
        Resources only consider events sent as requests, not generally published
        events.
        """

        pass

    def refresh(self):

        """
        Refresh messages are typically sent to event organisers, but resources
        do not act as organisers themselves.
        """

        pass

    def reply(self):

        "Since this handler does not send requests, it will not handle replies."

        pass

    def request(self):

        """
        Respond to a request by preparing a reply containing accept/decline
        information for the recipient.

        No support for countering requests is implemented.
        """

        self._process(self._schedule_for_attendee)

class Freebusy(CommonFreebusy, Handler):

    "A free/busy handler."

    def publish(self):

        "Resources ignore generally published free/busy information."

        self._record_freebusy(from_organiser=True)

    def reply(self):

        "Since this handler does not send requests, it will not handle replies."

        pass

    # request provided by CommonFreeBusy.request

# Handler registry.

handlers = [
    ("VFREEBUSY",   Freebusy),
    ("VEVENT",      Event),
    ]

# vim: tabstop=4 expandtab shiftwidth=4
