#!/usr/bin/env python

"""
Parsing of vCalendar and iCalendar files.

Copyright (C) 2008, 2009, 2011, 2013, 2014, 2015,
              2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.

--------

References:

RFC 5545: Internet Calendaring and Scheduling Core Object Specification
          (iCalendar)
          http://tools.ietf.org/html/rfc5545

RFC 2445: Internet Calendaring and Scheduling Core Object Specification
          (iCalendar)
          http://tools.ietf.org/html/rfc2445
"""

import vContent
import re

try:
    set
except NameError:
    from sets import Set as set

ParseError = vContent.ParseError

# Format details.

SECTION_TYPES = set([
    "VALARM", "VCALENDAR", "VEVENT", "VFREEBUSY", "VJOURNAL", "VTIMEZONE", "VTODO",
    "DAYLIGHT", "STANDARD"
    ])
QUOTED_PARAMETERS = set([
    "ALTREP", "DELEGATED-FROM", "DELEGATED-TO", "DIR", "MEMBER", "SENT-BY"
    ])
MULTIVALUED_PARAMETERS = set([
    "DELEGATED-FROM", "DELEGATED-TO", "MEMBER"
    ])
QUOTED_TYPES = set(["URI"])

unquoted_separator_regexp = re.compile(r"(?<!\\)([,;])")

# Parser classes.

class vCalendarStreamParser(vContent.StreamParser):

    "A stream parser specifically for vCalendar/iCalendar."

    def next(self):

        """
        Return the next content item in the file as a tuple of the form
        (name, parameters, value).
        """

        name, parameters, value = vContent.StreamParser.next(self)
        return name, self.decode_parameters(parameters), value

    def decode_content(self, value):

        """
        Decode the given 'value' (which may represent a collection of distinct
        values), replacing quoted separator characters.
        """

        sep = None
        values = []

        for i, s in enumerate(unquoted_separator_regexp.split(value)):
            if i % 2 != 0:
                if not sep:
                    sep = s
                continue
            values.append(self.decode_content_value(s))

        if sep == ",":
            return values
        elif sep == ";":
            return tuple(values)
        else:
            return values[0]

    def decode_content_value(self, value):

        "Decode the given 'value', replacing quoted separator characters."

        # Replace quoted characters (see 4.3.11 in RFC 2445).

        value = vContent.StreamParser.decode_content(self, value)
        return value.replace(r"\,", ",").replace(r"\;", ";")

    # Internal methods.

    def decode_quoted_value(self, value):

        "Decode the given 'value', returning a list of decoded values."

        if value[0] == '"' and value[-1] == '"':
            return value[1:-1]
        else:
            return value

    def decode_parameters(self, parameters):

        """
        Decode the given 'parameters' according to the vCalendar specification.
        """

        decoded_parameters = {}

        for param_name, param_value in parameters.items():
            if param_name in QUOTED_PARAMETERS:
                param_value = self.decode_quoted_value(param_value)
                separator = '","'
            else:
                separator = ","
            if param_name in MULTIVALUED_PARAMETERS:
                param_value = param_value.split(separator)
            decoded_parameters[param_name] = param_value

        return decoded_parameters

class vCalendarParser(vContent.Parser):

    "A parser specifically for vCalendar/iCalendar."

    def parse(self, f, parser_cls=None):
        return vContent.Parser.parse(self, f, (parser_cls or vCalendarStreamParser))

    def makeComponent(self, name, parameters, value=None):

        """
        Make a component object from the given 'name', 'parameters' and optional
        'value'.
        """

        if name in SECTION_TYPES:
            return (name, parameters, value or [])
        else:
            return (name, parameters, value or None)

# Writer classes.

class vCalendarStreamWriter(vContent.StreamWriter):

    "A stream writer specifically for vCalendar."

    # Overridden methods.

    def write(self, name, parameters, value):

        """
        Write a content line, serialising the given 'name', 'parameters' and
        'value' information.
        """

        if name in SECTION_TYPES:
            self.write_content_line("BEGIN", {}, name)
            for n, p, v in value:
                self.write(n, p, v)
            self.write_content_line("END", {}, name)
        else:
            vContent.StreamWriter.write(self, name, parameters, value)

    def encode_parameters(self, parameters):

        """
        Encode the given 'parameters' according to the vCalendar specification.
        """

        encoded_parameters = {}

        for param_name, param_value in parameters.items():
            if param_name in QUOTED_PARAMETERS:
                separator = '","'
            else:
                separator = ","
            if param_name in MULTIVALUED_PARAMETERS:
                param_value = separator.join(param_value)
            if param_name in QUOTED_PARAMETERS:
                param_value = self.encode_quoted_parameter_value(param_value)
            encoded_parameters[param_name] = param_value

        return encoded_parameters

    def encode_content(self, value):

        """
        Encode the given 'value' (which may be a list or tuple of separate
        values), quoting characters and separating collections of values.
        """

        if isinstance(value, list):
            sep = ","
        elif isinstance(value, tuple):
            sep = ";"
        else:
            value = [value]
            sep = ""

        return sep.join([self.encode_content_value(v) for v in value])

    def encode_content_value(self, value):

        "Encode the given 'value', quoting characters."

        # Replace quoted characters (see 4.3.11 in RFC 2445).

        value = vContent.StreamWriter.encode_content(self, value)
        return value.replace(";", r"\;").replace(",", r"\,")

# Public functions.

def parse(stream_or_string, encoding=None, non_standard_newline=0):

    """
    Parse the resource data found through the use of the 'stream_or_string',
    which is either a stream providing Unicode data (the codecs module can be
    used to open files or to wrap streams in order to provide Unicode data) or a
    filename identifying a file to be parsed.

    The optional 'encoding' can be used to specify the character encoding used
    by the file to be parsed.

    The optional 'non_standard_newline' can be set to a true value (unlike the
    default) in order to attempt to process files with CR as the end of line
    character.

    As a result of parsing the resource, the root node of the imported resource
    is returned.
    """

    return vContent.parse(stream_or_string, encoding, non_standard_newline, vCalendarParser)

def iterparse(stream_or_string, encoding=None, non_standard_newline=0):

    """
    Parse the resource data found through the use of the 'stream_or_string',
    which is either a stream providing Unicode data (the codecs module can be
    used to open files or to wrap streams in order to provide Unicode data) or a
    filename identifying a file to be parsed.

    The optional 'encoding' can be used to specify the character encoding used
    by the file to be parsed.

    The optional 'non_standard_newline' can be set to a true value (unlike the
    default) in order to attempt to process files with CR as the end of line
    character.

    An iterator is returned which provides event tuples describing parsing
    events of the form (name, parameters, value).
    """

    return vContent.iterparse(stream_or_string, encoding, non_standard_newline, vCalendarStreamParser)

def iterwrite(stream_or_string=None, write=None, encoding=None, line_length=None):

    """
    Return a writer which will either send data to the resource found through
    the use of 'stream_or_string' or using the given 'write' operation.

    The 'stream_or_string' parameter may be either a stream accepting Unicode
    data (the codecs module can be used to open files or to wrap streams in
    order to accept Unicode data) or a filename identifying a file to be
    written.

    The optional 'encoding' can be used to specify the character encoding used
    by the file to be written.

    The optional 'line_length' can be used to specify how long lines should be
    in the resulting data.
    """

    return vContent.iterwrite(stream_or_string, write, encoding, line_length, vCalendarStreamWriter)

def to_dict(node):

    "Return the 'node' converted to a dictionary representation."

    return vContent.to_dict(node, SECTION_TYPES)

to_node = vContent.to_node

# vim: tabstop=4 expandtab shiftwidth=4
