/*
 * Copyright (c) 2016 Vivid Solutions.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * and Eclipse Distribution License v. 1.0 which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v20.html
 * and the Eclipse Distribution License is available at
 *
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */
package test.jts.perf.simplify;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.io.WKTReader;
import org.locationtech.jts.simplify.TopologyPreservingSimplifier;

/**
 * Tests if TopologyPreservingSimplifier results are stable
 * over repeated calls with different instances of an identical MultiLineString.
 * 
 * This was a bug in GEOS, since the order of simplification depended on the 
 * (unstable) address of the geometry components.
 * (See https://github.com/libgeos/geos/pull/718).
 * 
 * It appears JTS is stable as it stands.
 * Although this is 
 * suspicious, since the order of HashMap would seem to be unstable?
 * 
 * @author mdavis
 *
 */
public class TopologyPreservingSimplifierStabilityTest {
  public static void main(String args[]) {
    try {
      (new TopologyPreservingSimplifierStabilityTest()).run();
    }
    catch (Exception ex) {
      ex.printStackTrace();
    }
  }

  // repro case from GEOS bug report
  String wkt = 
"MULTILINESTRING((159612.02323361888 6284563.8677484,159743.35796885678 6284436.870843225,159804.50576514952 6284376.057057197,159943.10966313622 6284232.942543113,160062.97849082242 6284104.710601479,"
+"160212.85905322648 6283940.435912305,160278.87151126692 6283866.104831385,160358.70985006384 6283774.160091971,160552.72859056742 6283538.077492254,160684.97614562983 6283370.620643681,"
+"160811.55753861085 6283201.910250149,161138.15779264923 6282756.753832039,161584.226124207 6282147.686750394,161704.9855078195 6281984.995507281,161836.4315625482 6281805.648176685,"
+"161986.86872240627 6281612.872298865,162079.47540679717 6281496.176424808,162250.50667245197 6281295.324729303,162338.95000788724 6281195.4440528685,162430.53255296283 6281096.515461827,"
+"162580.01236520006 6280943.94763531,162596.9329278006 6280927.239206045,162672.95300806337 6280852.187522901,162769.03286056704 6280761.03967867,162860.51521810095 6280675.92056996,"
+"163020.51472221813 6280534.855322192,163099.85212330648 6280467.05722754,163230.44101795607 6280360.683402378,163363.5346011485 6280256.823770906,163467.76304037822 6280179.112405164,"
+"163648.79079630628 6280051.332500253,163770.30715245623 6279968.682376769,163963.03458686662 6279844.963376556,164159.33537693147 6279726.662066799,164358.04066799747 6279615.085483966,"
+"164543.98874541855 6279518.976647851,164709.18686975574 6279437.961534301,164812.15739873954 6279390.289969728,164916.3747060202 6279343.246819833,165085.1461860119 6279271.332933188,"
+"165186.24654755034 6279231.420650729,165324.78365384255 6279176.721946358,165504.15274935775 6279112.499804153,165669.88520725077 6279057.46234304,166039.34346524457 6278943.365158198,"
+"166557.42437539648 6278795.793149981,166583.84049056174 6278788.238917756,167189.73021505136 6278618.109211654,167769.8717413205 6278455.129456483,167928.78031442792 6278413.489575424,"
+"168027.06429284927 6278389.860349767,168103.07324116293 6278372.376123329,168372.9228187949 6278316.477766411,168581.99195445376 6278280.134696615,168742.52579212672 6278256.30214721,"
+"168872.51356152602 6278239.7348305425,169075.40446544587 6278218.839009227,169235.6711363409 6278205.5648425175,169378.816869552 6278196.09300741,169525.63614595926 6278190.508364088,"
+"169700.730573028 6278186.570259162,169851.34584407127 6278186.315640371,170044.14007017616 6278190.372567335,170194.27666740905 6278198.07903647,170293.64044489112 6278203.986202594,"
+"170401.4199758772 6278212.354695007,170499.50357921512 6278220.723195708,171085.58956629265 6278281.98494721,171705.6279980621 6278347.032546713,171877.6166113377 6278364.924119366,"
+"172140.3417415589 6278387.178301031,172250.04709973568 6278395.292349814,172467.320481866 6278410.0096924985,172565.3706893567 6278415.8321315665,172732.4612450374 6278421.960125714,"
+"172873.4807759743 6278427.290295983,173072.53115746178 6278431.551039599,173317.93497491552 6278433.367373186,173696.47690335807 6278430.8211112525,174099.07497376102 6278417.376860977,"
+"174673.76184498132 6278398.178101262,174785.28171085802 6278392.304749357,174919.56641260194 6278379.845108834,174999.9724808019 6278368.6416330505,175050.34455038584 6278360.748284028,"
+"175121.71147593344 6278348.424489337,175222.12165662897 6278327.290751028,175329.82326397148 6278301.030669535,175465.9781331607 6278262.582798295,175584.2884879758 6278220.960840882,"
+"175688.11617703867 6278181.596706854,175823.68105292672 6278123.068570675,175954.45919071068 6278060.959254745,176051.79695346032 6278008.423854744,176149.13471620996 6277951.679200016,"
+"176247.7081253074 6277886.770436193,176332.1439590741 6277828.889349307,176413.15115252437 6277768.157069202,176490.89668489437 6277706.525623646,176554.3265307484 6277652.481784668,"
+"176627.05155408365 6277584.842605183,176657.0632888015 6277557.413671906,176711.01984598898 6277504.507992098,176760.8241861699 6277453.130223573,176820.9701070455 6277390.431777683,"
+"177286.2855785614 6276892.20443966,177861.65149867552 6276284.154003537,178398.8570973457 6275710.3070790805,178695.34542912449 6275395.866013061,178775.66244173184 6275316.536399705,"
+"178858.2392400023 6275241.313948316,178944.2780744364 6275168.569570697,179282.555743059 6274899.63970496,179341.2990383506 6274851.1623121975,179428.87408175768 6274776.724518886,"
+"179508.23374674423 6274703.390274527,179556.8135725264 6274653.3699657265,179603.24493213627 6274605.046692117,179711.58106057628 6274482.339396265,179772.46169009112 6274415.709673763,"
+"179860.53767120678 6274326.463656855,179922.175273259 6274271.881528828,179987.39736291478 6274218.419549947,180033.8621183719 6274184.910548036,180080.6942281486 6274154.099350101,"
+"180113.37763064553 6274134.21464984,180161.62349795533 6274106.525452321,180227.2797336252 6274070.115652964,180274.1897670455 6274047.771040161,180324.66202417118 6274025.0023307055,"
+"180385.19756326458 6274002.216716409,180442.14861475438 6273980.992048297,180512.50253293573 6273958.885196452,180622.23015501068 6273925.954068508,180758.819170214 6273887.695757958,"
+"181342.9348023045 6273730.813046822,181445.46005332508 6273703.7698630225,182199.37130472253 6273502.49257374,182963.31244224045 6273298.794114928,183938.78287616375 6273037.642466021,"
+"184079.81353904973 6272997.267655166,184238.04306326332 6272948.394035007,184438.81889685805 6272888.1548454845,185007.9731893859 6272716.159139401,185901.1673876639 6272446.119879661,"
+"185976.12993276407 6272423.898300745,186121.39073630024 6272380.829315267,186199.8821092586 6272357.556209243,186287.8356389343 6272331.484299947,186334.95717938713 6272317.506939917,"
+"186380.0304412093 6272304.157225641,186534.40831104142 6272258.391783233,187140.94368857768 6272080.624735446,187160.71403014255 6272073.754996126,187539.9238755298 6271955.324960422,"
+"187600.55960216493 6271936.378344127,187674.38668845902 6271913.293043615,188332.00658032027 6271715.4507022835,188782.3273164263 6271579.470701108,188845.1115092337 6271560.0669725705,"
+"189141.16569499843 6271470.308346317,189362.03469668137 6271405.008357309,189660.70489047974 6271327.564920277,189695.74826618144 6271318.185594663,190457.64112506877 6271112.86068684,"
+"190816.96930940036 6271015.305118234,190957.20960390175 6270978.976471972,191363.30310631558 6270868.855156802,191382.17176000506 6270863.7502299575,191434.1802261037 6270849.63961765,"
+"191820.4365952582 6270744.9131529685,191901.21001777778 6270719.338045002,191985.0892540905 6270686.9791996395,192040.4595688111 6270663.354607214,192160.21707700647 6270601.4866406545,"
+"192224.78238166656 6270561.2591377525,192288.2122275206 6270517.639987497,192333.53039222254 6270484.077816298,192375.79840287674 6270448.616364298,192436.43412951185 6270396.28089196,"
+"192495.3110081924 6270340.316552183,192562.12496656654 6270270.209020157,192607.55445075926 6270222.555224079,192756.95633935288 6270062.772818872,192822.64597087 6269995.906593106,"
+"192877.8047785581 6269946.151545336,192950.19584342095 6269885.764165967,193028.06382723083 6269828.853568001,193081.4526550153 6269795.378919636,193143.23497240557 6269758.885938274,"
+"193203.2473098922 6269726.344235049,193277.89816041818 6269691.106408963,193383.50696133377 6269650.425386225,193473.76480446896 6269622.140405321,193562.74247346004 6269598.857935085,"
+"193760.70192393768 6269560.042655652,193859.23080523883 6269538.235634942,193947.90791160477 6269513.155934936,194042.86343725142 6269480.309949474,194138.18631721768 6269439.375585526,"
+"194224.10270021195 6269397.305304478,194320.43858754443 6269340.66930346,194475.90738838635 6269233.418035558,194494.54227114512 6269220.615801189,194664.23770291038 6269112.128240871,"
+"194828.89036174273 6269037.198367567,195010.69735410626 6268978.30961655,195148.03220989797 6268949.297773971,195257.4258735005 6268935.122532327,195372.96437299484 6268930.30702502,"
+"195454.81759457514 6268931.917845822,195574.4415193816 6268933.816919159,195727.56147896775 6268938.327220044,195820.25721895127 6268939.463273653,195941.77357510122 6268941.85407354,"
+"195965.09500842242 6268942.464490643,196037.35248989632 6268942.277974303,196407.6901718674 6268913.825802466,196571.5524623151 6268895.682930008,196777.5714439262 6268865.060562559,"
+"197081.7074247225 6268811.0732258195,197359.3048389137 6268768.921292356,197475.16616493132 6268757.28970169,197670.76564220418 6268743.233471346,198172.98352491803 6268730.66935761,"
+"198995.62342993126 6268720.665555703,199347.37075693987 6268728.414262247,199766.67788291088 6268759.137868818,200210.07454668958 6268814.12526456,200531.89919457294 6268876.590566079,"
+"200935.03159853164 6268964.35475524,201114.24484675977 6268993.722728287,201231.23049963443 6269013.934464141,201370.8140091401 6269028.9067794895,201490.69396877536 6269036.316644114,"
+"201616.45159752454 6269038.826164957,201729.85276279564 6269034.231029387,201869.69230713014 6269026.32943679,202013.5616170314 6269008.321972242,202197.13858929859 6268977.800933472,"
+"202565.27214535192 6268895.954225012,202962.90536646548 6268805.5965146255,203348.71645765682 6268718.393507428,203397.96420038375 6268706.863719617,203489.92523172806 6268685.499742586,"
+"203619.6903621458 6268658.896579973,203844.1883792286 6268616.202833018,204131.44832522067 6268588.548658428,204371.29730008382 6268588.718211642,204675.25516969484 6268610.488872914,"
+"204941.39780828342 6268652.775660202,205011.76285841386 6268673.444379371,205136.97502165815 6268706.69416403,205236.40559083468 6268736.841200623,205346.28906019675 6268772.973701606,"
+"205514.38149129457 6268842.237984302,205880.70053964798 6269000.77648296,206415.36805392808 6269237.284144589,206920.04609738846 6269441.325643939,207099.98292230672 6269503.846450623,"
+"207507.6237656426 6269626.668030797,207670.59550016397 6269669.010772909,207843.78636394013 6269708.640515808,208195.21086442543 6269775.453596626,208326.70144695044 6269795.412835121,"
+"208401.7530476433 6269804.53610511,208587.40056243923 6269826.15727683,208844.6265098152 6269851.678809498,209191.79860575218 6269865.550345944,209374.51841794027 6269862.04006501,"
+"210445.4453152188 6269847.066274299,210891.858737198 6269838.536482385,211903.04046376777 6269819.865934097,211997.6842948402 6269818.136239613,212192.64925101554 6269814.931218413,"
+"212407.40681265394 6269808.080277704,212479.0297730303 6269801.093680517,212650.52858054647 6269780.625697236,212924.0071735783 6269723.664934162,213171.13644313932 6269647.220428596,"
+"213310.74221654324 6269590.3114362415,213451.52797654943 6269524.686822251,213657.66940960046 6269410.09084941,213828.21086949573 6269294.6825062595,213881.0430998262 6269257.089502756,"
+"214002.68190741603 6269159.029694296,214111.9642515278 6269056.952388293,214243.57728549268 6268925.338916485,214374.81183318887 6268788.657730618,214873.33390880836 6268207.147302767,"
+"215218.01244815157 6267810.013090632,215882.93489860886 6267037.414654649,216116.60564173307 6266750.062595269,216274.74611035397 6266549.306755744,216459.60325676532 6266299.9747393485,"
+"216683.26637766714 6265986.527581096,216817.5844752583 6265785.0440774225,217058.769284011 6265410.989382424,217213.72601519522 6265167.233065622,217368.43784349976 6264940.143649308,"
+"217465.0075017629 6264806.035119937,217572.3306228367 6264677.623069013,217658.02436684936 6264585.649580556,217711.70262530987 6264529.503367279,217716.7120023956 6264524.385318675,"
+"217819.73819112475 6264423.8213760685,218037.39005952372 6264242.168710179,218235.13800296892 6264110.545412234,218316.47915489157 6264061.960376094,218399.21180044912 6264015.256642067,"
+"218491.45113052044 6263968.7056804495,218519.58156584387 6263954.80992066,218626.33695751466 6263902.701024731,218868.96891964765 6263798.060552794,219003.18682969714 6263742.902412033,"
+"219524.27336610042 6263522.578458384,219917.03079351725 6263360.025015918,220398.06457713316 6263157.553507509,220812.79647203258 6262984.230630194,221226.7046027001 6262802.134449159,"
+"221473.62236522866 6262688.324208363,221777.62476263603 6262540.71374249,221856.95103177536 6262498.677762082,221929.31983274 6262464.012213491,222114.2103749986 6262367.962615812,"
+"222255.61952415327 6262288.670419796,222381.59979188404 6262211.056278395,222434.3429666219 6262170.733124265,222484.49239722424 6262128.885348189,222542.9351298907 6262075.178181773,"
+"222597.90469444444 6262018.862263656,222674.04722614703 6261927.527586906),(212490.06153456794 6198000.715026029,212379.49901631207 6197893.050431567,212340.8711530068 6197854.040638241,212310.31395278405 6197824.245347578,212241.1066253579 6197762.149741283,"
+"212189.75494425494 6197720.618408016,212145.14922429406 6197687.427115726,212111.6309256162 6197664.038597945,212062.349787042 6197631.318320755,212029.2545024292 6197610.889228954,"
+"211996.00337052924 6197590.880536372,211946.154502552 6197563.8268411225,211885.16255354637 6197533.5785913365,211819.5842415201 6197503.4481459,211775.4015356242 6197485.4573616,"
+"211727.4896267868 6197467.0294574825,211667.66653243446 6197446.92022078,211615.80278167388 6197431.636560121,211558.41758416998 6197414.469796878,211474.1153337922 6197390.611248074,"
+"211228.5556690513 6197324.248180438,210481.3347190506 6197120.454419101,209765.6617127406 6196925.240006449,209577.96591931404 6196872.246564336,209373.60559811577 6196811.015339376,"
+"209173.5422092621 6196747.19545404,208980.7702470554 6196679.374741058,208855.96996592707 6196631.56095884,208734.5204014716 6196583.814693298,208336.28605510772 6196426.169858823,"
+"208073.10451497426 6196315.784907852,207715.27914376836 6196180.6720451135,207378.07014225743 6196062.170566912,206998.59313009222 6195938.5266493885,206917.69725613273 6195916.252443212,"
+"206501.23990912602 6195782.726103228,206074.01796735957 6195639.6536853695,205728.7828305624 6195515.713326804,205494.16587176654 6195425.326928047,205105.83896008326 6195256.49167308,"
+"205032.43488785418 6195226.503762168,204922.08387663082 6195178.126626321,204769.09750043362 6195112.066561723,204479.92285919993 6194989.764344907,204439.89237031064 6194973.157197529,"
+"204396.65588008653 6194955.844112794,204320.33523719865 6194926.243862112,204097.5515402741 6194843.007365857,204040.51143319166 6194821.66049573,203908.72028804146 6194776.042245602,"
+"203860.04027471755 6194759.502748834,203805.86107854848 6194742.34137406,203665.11984633855 6194695.412502371,203435.89075089706 6194625.322226611,203163.3138457406 6194546.509174163,"
+"202803.61830708943 6194456.637300373,202659.47069846117 6194421.677112061,202233.5177988898 6194319.839574057,201938.05360642634 6194250.307034981,201636.93438383052 6194178.287587824,"
+"201488.12248853804 6194143.177277796,201408.89640694048 6194124.672572378,201340.34586451 6194105.764536182,201184.33159816323 6194069.242732725,201132.23407647194 6194055.343311639,"
+"201081.4835206193 6194042.704438166,201023.70870489758 6194027.695800286,200964.11938147596 6194012.888872159,200889.20136417207 6193991.880221502,200718.41500139705 6193941.896648272,"
+"200622.92514219458 6193909.728491724,200503.95800238382 6193874.636095934,200456.98117726902 6193859.07313529,200414.7242985639 6193844.804311923,200306.24345478584 6193806.552554789,"
+"200149.65032708697 6193748.923145034,200057.88967082606 6193713.881831132,199998.3226113026 6193691.478994754,199907.66401800056 6193654.858131102,199882.28317409966 6193643.648380099,"
+"199852.04880040025 6193630.758025001,199781.7839378115 6193599.565799282,199765.854118679 6193592.776127363,199526.86230389492 6193485.284914192,199499.21054238186 6193472.7309163585,"
+"199394.03588748036 6193424.985559544,199020.4142805309 6193258.156209193,198857.6985808384 6193193.65713112,198534.29319618578 6193069.365861583,198394.4425199022 6193021.286667081,"
+"198375.74084544892 6193015.421729742,198159.32462339773 6192945.244454814,197966.2409666168 6192883.638157201,197677.7453742769 6192795.531507291,197673.94937964086 6192794.6240622755,"
+"197522.10959419888 6192747.537881321,197285.61133600856 6192673.195347736,197012.8340557687 6192588.2500511715,196679.29859745392 6192495.844646442,196342.17865153556 6192413.388078894,"
+"196240.06528263088 6192391.391954001,196167.58516217538 6192376.537460241,195831.57841116496 6192314.817652284,194658.80531175964 6192153.185458943,194042.79664555693 6192070.866634535,"
+"193426.9549585904 6191979.256582095,192988.79031087898 6191898.6701241145,192768.21073987213 6191854.075693898,192551.63867053378 6191805.583296708,192222.4001445636 6191732.811613842,"
+"191578.17198744477 6191587.639761255,190985.30664337796 6191453.643633212,190757.43564572415 6191397.979153161,190516.66271908736 6191332.637280303,190274.14207644516 6191260.004051219,"
+"190008.6339589541 6191166.638582807,189832.66010790807 6191103.045809532,189506.87248615248 6190971.846262171,189370.4170543381 6190911.027078318,189254.22176984805 6190857.684643403,"
+"189119.83688056245 6190793.237347289,188957.42174349504 6190710.242851276,188742.1298483009 6190594.320718837,188531.26846884025 6190474.384973415,188113.51981574032 6190218.475056823),("
+"188204.74613844542 6190253.283595123,188541.92174410916 6190454.426646578,188750.4565462122 6190576.176527881,188965.8041011518 6190693.42564406,189127.42886983452 6190774.471112138,"
+"189260.91207124476 6190839.825502848,189377.39678641086 6190893.184626676,189514.19730864666 6190953.8356740605,189839.0164508324 6191085.774195124,190014.7342670496 6191150.744545293,"
+"190279.73031488297 6191241.219985933,190538.85982555154 6191318.658317842,190810.07863292028 6191390.653580105,190989.77055495876 6191437.026580091,191582.42439199306 6191569.291828281,"
+"192261.58460532286 6191718.22710756,192557.19351312437 6191786.176333629,192775.04575660682 6191835.189502212,192992.80894449662 6191879.380567987,193430.10530017983 6191959.126698984,"
+"194044.73360469672 6192050.988580879,194660.00756226023 6192134.298603812,195836.8326911304 6192296.182491913,196171.15851782984 6192357.71732437,196246.9002993656 6192373.159914935,"
+"196347.23255641756 6192394.651880307,196685.8107876653 6192475.612718563,197017.70984946546 6192570.168827414,197292.43522079417 6192654.576207245,197526.96312399744 6192729.002600291,"
+"197682.89946670068 6192776.592813748,197961.43196461452 6192863.690987247,198096.13968042345 6192903.030817104,198165.8479455582 6192924.893825473,198382.62038997991 6192996.7514260365,"
+"198540.39350428124 6193050.493779081,198873.33896929485 6193177.994583501,199030.2215276698 6193242.980900124,199395.08229069386 6193404.650589195,199538.8402811043 6193469.82350049,"
+"199773.4795037983 6193575.297788921,199789.36479513452 6193581.331172849,199820.723495691 6193595.36426899,199891.18873336315 6193625.817007487,199915.07789608737 6193635.967944528,"
+"200004.9461210048 6193670.303074583,200064.95845849146 6193695.092351473,200159.0122962627 6193730.92348808,200310.99679704275 6193786.149447056,200422.96194088264 6193826.165802465,"
+"200463.87185374915 6193839.611071189,200510.74849132219 6193855.42609571,200628.39092919254 6193892.753703668,200721.8436417135 6193922.316747496,200893.4649006695 6193972.56911555,"
+"200968.67234864942 6193992.670145828,201027.8497899551 6194008.014860586,201083.55406314804 6194022.8553941585,201138.41230821097 6194037.258971866,201189.69719761945 6194049.763381143,"
+"201411.9465609882 6194103.663645866,201493.031678082 6194122.97504919,201642.9790321806 6194159.31220363,201941.86073301145 6194230.306240591,202255.09151620558 6194300.326034044,"
+"202664.54686724136 6194400.39852927,202837.5930156795 6194439.106764922,203168.4679381644 6194523.263749096,203445.46422710526 6194602.026162016,203544.5163100131 6194632.66739151,"
+"203672.2442937493 6194674.956839473,203865.63964510447 6194740.862236623,204047.00135950488 6194802.280273382,204104.97655030998 6194824.333053703,204333.71583999204 6194906.039775093,"
+"204468.23431266664 6194958.197345543,204544.04288589687 6194988.957519183,204614.67510280517 6195017.784777341,204737.99483470593 6195070.144740824,204851.76335429668 6195121.025822879,"
+"204968.28146531 6195172.478722892,205110.74814962727 6195237.900500616,205503.0602990809 6195403.625594119,205734.4378606947 6195496.432383454,206084.40407585062 6195621.465118841,"
+"206507.6630437448 6195761.897998581,206924.91075913614 6195893.927863353,207384.72704780684 6196043.443147057,207720.80059051173 6196161.860308602,208083.39043592356 6196298.485903498,"
+"208236.52152745883 6196359.646195032,208345.10255877854 6196404.499596358,208731.15855284967 6196561.471550474,208811.66480859133 6196592.422450956,208985.97999922454 6196659.855791187,"
+"209181.13419853422 6196729.021332928,209374.61860548198 6196791.294338468,209579.6579755741 6196852.542234097,209771.3835345674 6196905.720492717,210481.1343439671 6197100.177870892,"
+"211234.67824104498 6197305.4843564965,211596.0992318035 6197403.002889102,211707.1070280225 6197433.032096752,211758.9262509868 6197446.735269826,211853.89290858255 6197472.6452561505,"
+"211909.9088763497 6197488.736053101,211991.0162573417 6197515.049699872,212032.62748300022 6197529.0388410585,212078.90299532298 6197546.979275725,212142.61113990398 6197573.663009996,"
+"212210.04848742654 6197605.760946121,212285.33385905004 6197644.584646139,212343.40923739687 6197677.574001187,212396.99844026475 6197709.974985597,212446.14599544997 6197742.981408434,"
+"212509.27527867886 6197788.497862243,212569.9666650594 6197837.024353374,212630.29069712025 6197885.43341908),("
+"196631.9766819177 6268913.673198705,196283.42422429487 6268957.199294584,196174.40904696102 6268969.509403638,195991.07697757357 6268986.5333305355,195876.45129790375 6268988.245898784,"
+"195746.86427867127 6268985.566831778,195721.3387194324 6268983.786439614,195652.521010224 6268976.308796625,195572.19286566757 6268967.050771607,195481.43408482382 6268961.370486556,"
+"195393.74772192593 6268960.709199883,195302.0204615123 6268964.507359918,195173.55776913685 6268976.902260108,195047.6220292024 6268998.402622558,194880.38675818368 6269047.372106385,"
+"194732.28730763233 6269108.8726181835,194617.03823881404 6269169.780100242,194446.2407440899 6269280.862776473,194310.3864375258 6269374.922168917),("
+"196631.9766819177 6268913.673198705,196220.71795513103 6268983.0573267415,196106.35944223907 6268998.402622558,195991.5445194349 6269007.931980497,195870.78513582237 6269011.831898765,"
+"195757.01661623162 6269010.05150107,195649.23708524558 6269002.845133285,195504.9224973812 6268987.109838621,195374.35586662975 6268981.870398944,195278.09790294082 6268987.957644698,"
+"195175.8398186981 6268996.622227694,195123.84248454857 6269004.845959488,195079.53732721283 6269013.103611632,194983.84709292694 6269036.164038138,194885.75235763993 6269066.3292057,"
+"194730.49506383055 6269133.069385506,194622.70440089543 6269189.195300187,194523.87495696914 6269247.1189913,194310.3864375258 6269374.922168917),("
+"222375.4104281959 6262244.687291328,222306.75969822367 6262289.144815342,222214.37565281437 6262342.988882456,222115.24564626295 6262398.307325896,221924.88931700645 6262495.356911591,"
+"221788.23351010866 6262561.994431308,221749.14923689113 6262582.597442175,221482.4833966958 6262710.0796876205,221236.01091213044 6262824.262984011,220819.9988430869 6263007.342401228,"
+"220407.61578944323 6263180.818252401,219927.29445056838 6263381.409435056,219530.9191397008 6263545.623919324,219012.6601183636 6263766.219603694,218879.58879906934 6263822.32685951,"
+"218705.75228224657 6263893.838307869,218520.03797575616 6263979.347814357,218490.1486924782 6263994.362057897,218408.25094310154 6264037.557812361,218325.8745199145 6264082.9229123965,"
+"218244.90072231152 6264132.050353388,218053.4089342489 6264260.657482187,217837.0038441468 6264442.852849489,217784.0825582237 6264493.355096953,217728.20017384543 6264546.501411833,"
+"217658.1579502383 6264620.205192584,217541.57304753052 6264753.565079774,217384.59030161385 6264956.820506265,217233.96389862147 6265182.18162379,217114.51808500028 6265370.05776213,"
+"217054.32763632835 6265464.734680305,216838.23424080043 6265799.298788547,216701.3669268701 6265999.426621184,216479.58510536267 6266315.416871801,216294.72795895135 6266564.749343449,"
+"216136.38711524702 6266765.352984541,215902.29335805785 6267053.485366645,215235.44508040982 6267830.493258173,214892.12463885426 6268223.8306030845,214377.20520224093 6268825.044789955,"
+"214251.2694623065 6268952.722895732,214128.18350133637 6269077.791658462,214009.2608893219 6269184.023561346,213892.23070865095 6269278.014049706,213843.42824388717 6269314.742350535,"
+"213670.8273734122 6269430.0661824485,213462.23691156376 6269544.865912889,213305.5324643741 6269618.477608959,213177.59297360538 6269668.671623153,212927.60279313088 6269747.659969178,"
+"212654.14646399723 6269802.8233715175,212483.09293444428 6269824.614116857,212408.41982002018 6269831.159830317,212191.34681297326 6269838.383861936,211998.31881593776 6269841.673681193,"
+"211873.9415488744 6269845.557026199,210892.30401516115 6269862.04006501,210446.35813504332 6269870.315512252,209374.51841794027 6269887.171673696,209190.60748720073 6269888.273938942,"
+"208845.5170657416 6269873.89362822,208584.40606813686 6269851.678809498,208322.51583409662 6269819.798102936,208119.68058992218 6269787.374869348,207838.4764242293 6269733.17815191,"
+"207666.2651719721 6269693.32784602,207500.0763041668 6269651.188471549,207094.13864904008 6269527.942605048,207010.21488493102 6269499.573248141,206915.03672030274 6269469.050436069,"
+"206405.47175119657 6269260.463879191,205872.1289388569 6269025.413802074,205509.60588513955 6268865.823577013,205338.81952236447 6268798.356409601,205229.83774087788 6268760.79952401,"
+"205129.21605314984 6268728.482084657,205004.1708691418 6268697.33470097,204937.22332737868 6268683.159881713,204672.08256420726 6268635.854137745,204370.6739109354 6268613.6256161155,"
+"204261.52515021258 6268613.47301777,204135.43356299106 6268617.254066456,203986.54374405506 6268624.867035082,203844.71158083534 6268640.4999231035,203624.3657807591 6268684.635011318,"
+"203495.53573406406 6268709.915720742,203397.38533903164 6268732.63620819,203350.37511806964 6268743.369116401,202970.9426337008 6268826.943839294,202600.93891020207 6268911.401098567,"
+"202196.83802667345 6268999.894763305,202016.15536116686 6269030.53457528,201871.68492601535 6269047.066894041,201731.35557592136 6269057.528901179,201616.9970630294 6269060.801460366,"
+"201488.60116234847 6269057.8171576345,201368.87705000027 6269051.84855534,201231.82049293566 6269038.114003561,201055.81324604238 6269011.509731532,200931.7922013496 6268991.348869861,"
+"200526.58925486208 6268901.057963869,200204.19687757565 6268843.136643355,199764.70752792383 6268786.250018328,199346.39114542087 6268756.679297957,198995.5343743386 6268745.810727753,"
+"198173.98540033516 6268752.711674767,197670.8101700005 6268765.411466662,197477.55953398338 6268779.162531202,197360.328978229 6268791.52324825,197084.12305767272 6268832.776636343,"
+"196819.97303796935 6268881.422998699,196631.9766819177 6268913.673198705))"
      ;
  
  private void run() {
    double prevLen = -1;
    for (int i = 0; i < 1000; i++) {
      double len = simplify();
      System.out.println("len: " + len);
      if (prevLen >= 0 && len != prevLen) {
        System.out.println("Found diff in len: " + len);
        throw new IllegalStateException("Found diff in len: " + len);
      }
      prevLen = len;
    }
    
  }

  private double simplify() {
    Geometry geom = read(wkt);
    Geometry simp = TopologyPreservingSimplifier.simplify(geom, 1222);
    double simpLen = simp.getLength();
    return simpLen;
  }

  private static Geometry read(String wkt) {
    WKTReader rdr = new WKTReader();
    try {
      return rdr.read(wkt);
    }
    catch (Exception ex) {
      ex.printStackTrace();
    }
    return null;
  }
}
