# Copyright (c) 2015 Red Hat, Inc.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from castellan import key_manager as km
try:
    from castellan.key_manager import barbican_key_manager as bkm
except ImportError:
    bkm = None


def set_defaults(conf, api_class=None, barbican_endpoint=None,
                 barbican_api_version=None):
    '''Set defaults for configuration values.

    Overrides the default options values.

    :param conf: Config instance in which to set default options.

    :param api_class: The full class name of the key manager API class.

    :param barbican_endpoint: Use this endpoint to connect to Barbican.

    :param barbican_api_version: Version of the Barbican API.
    '''
    conf.register_opts(km.key_manager_opts, group='key_manager')
    if bkm:
        conf.register_opts(bkm.barbican_opts, group=bkm.BARBICAN_OPT_GROUP)

    if api_class is not None:
        conf.set_default('api_class', api_class, group='key_manager')
    if bkm is not None and barbican_endpoint is not None:
        conf.set_default('barbican_endpoint', barbican_endpoint,
                         group=bkm.BARBICAN_OPT_GROUP)
    if bkm is not None and barbican_api_version is not None:
        conf.set_default('barbican_api_version', barbican_api_version,
                         group=bkm.BARBICAN_OPT_GROUP)


def list_opts():
    '''Returns a list of oslo.config options available in the library.

    The returned list includes all oslo.config options which may be registered
    at runtime by the library.

    Each element of the list is a tuple. The first element is the name of the
    group under which the list of elements in the second element will be
    registered. A group name of None corresponds to the [DEFAULT] group in
    config files.

    The purpose of this is to allow tools like the Oslo sample config file
    generator to discover the options exposed to users by this library.

    :returns: a list of (group_name, opts) tuples
    '''
    opts = [('key_manager', km.key_manager_opts)]
    if bkm is not None:
        opts.append((bkm.BARBICAN_OPT_GROUP, bkm.barbican_opts))
    return opts
