/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iorthotextureplanefilter.h"


#include "idatalimits.h"
#include "ierror.h"
#include "iorthoslicer.h"
#include "imath.h"

#include <vtkCellArray.h>
#include <vtkCellData.h>
#include <vtkPointData.h>
#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "igenericfiltertemplate.h"


iOrthoTexturePlaneFilter::iOrthoTexturePlaneFilter(iViewSubject *vo) : iGenericFilter<vtkStructuredPointsToPolyDataFilter,vtkImageData,vtkPolyData>(vo,1,true,false) 
{
	mTexture = 0;
	mOffX = mOffY = 0;
}


void iOrthoTexturePlaneFilter::SetTexturePiece(vtkImageData *texture, int offx, int offy, double globalOrg[3], int globalDims[3])
{
	int i;

	mTexture = texture;
	mOffX = offx;
	mOffY = offy;

	for(i=0; i<3; i++)
	{
		mGlobalOrg[i] = globalOrg[i];
		mGlobalDims[i] = globalDims[i];
	}

	this->Modified();
}


void iOrthoTexturePlaneFilter::ProduceOutput()
{
	int dims[3], dimsTexture[3];
	double org[3], spa[3];

	vtkImageData *input = this->GetInput();
	vtkPolyData *output = this->GetOutput();

	output->Initialize();

	if(mTexture == 0) return;

	//
	//  Do we have cell or point data?
	//
	vtkFloatArray *scalars;
	bool isPointData;
	if(input->GetPointData()->GetScalars() != 0)
	{
		isPointData = true;
		scalars = vtkFloatArray::SafeDownCast(input->GetPointData()->GetScalars());
	}
	else if(input->GetCellData()->GetScalars() != 0)
	{
		isPointData = false;
		scalars = vtkFloatArray::SafeDownCast(input->GetCellData()->GetScalars());
	}
	else
	{
		//
		//  Must be quiet here for parallel execution
		//
		return;
	}

	if(scalars == 0) return;

	//
	//  Prepare texture support
	//
	input->GetOrigin(org);
	input->GetSpacing(spa);
	input->GetDimensions(dims);

	int i, Axis = -1;
	for(i=0; i<3; i++)
	{
		if(dims[i] == 1) Axis = i;
	}
	if(Axis == -1) return;

    int u, v, dimU, dimV;
	iOrthoSlicer::GetUV(Axis,u,v);

	if(isPointData)
	{
		dimU = dims[u] - 1;
		dimV = dims[v] - 1;
	}
	else
	{
		dimU = dims[u];
		dimV = dims[v];
	}

	if(dimU<1 || dimV<1)
	{
		return;
	}

	mTexture->GetDimensions(dimsTexture);

	//
	//  AppendPolyDataFilter does not merge texture coordinates into one, so we need to create the correct texture support here.
	//  We create it for zero offset instance, and skip it altogether for other instances.
	//
	if(mOffX==0 && mOffY==0)
	{
		float tc[2], pad;
		static vtkIdType pts[4]={0,1,2,3};

		int dimU, dimV;
		if(isPointData)
		{
			dimU = mGlobalDims[u] - 1;
			dimV = mGlobalDims[v] - 1;
			pad = 0.0;
		}
		else
		{
			dimU = mGlobalDims[u] - 1;
			dimV = mGlobalDims[v] - 1;
			pad = 0.5;
		}
		//
		// We'll create the building blocks of polydata including data attributes.
		//
		vtkPoints *points;
		points = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(points);

		vtkCellArray *polys;
		polys = vtkCellArray::New(); IERROR_ASSERT(polys);

		vtkFloatArray *tcoords;
		tcoords = vtkFloatArray::New(); IERROR_ASSERT(tcoords);

		points->SetNumberOfPoints(4);
		tcoords->SetNumberOfComponents(2);
		tcoords->SetNumberOfTuples(4);

		//
		// Load the cell, and data attributes.
		//
		polys->InsertNextCell(4,pts);

		//
		//  Place the support plane
		//
		double x1[3];
		x1[Axis] = org[Axis];
		//
		//  LL point 
		//
		x1[u] = mGlobalOrg[u] - pad*spa[u];
		x1[v] = mGlobalOrg[v] - pad*spa[v];
		points->SetPoint(0,x1);
		tc[0] = 0.0;
		tc[1] = 0.0;
		tcoords->SetTuple(0,tc);
		//
		//  LR point 
		//
		x1[u] = mGlobalOrg[u] - pad*spa[u] + spa[u]*dimU;
		x1[v] = mGlobalOrg[v] - pad*spa[v];
		points->SetPoint(1,x1);
		tc[0] = float(dimU)/dimsTexture[0];
		tc[1] = 0.0;
		tcoords->SetTuple(1,tc);
		//
		//  UR point 
		//
		x1[u] = mGlobalOrg[u] - pad*spa[u] + spa[u]*dimU;
		x1[v] = mGlobalOrg[v] - pad*spa[v] + spa[v]*dimV;
		points->SetPoint(2,x1);
		tc[0] = float(dimU)/dimsTexture[0];
		tc[1] = float(dimV)/dimsTexture[1];
		tcoords->SetTuple(2,tc);
		//
		//  UL point 
		//
		x1[u] = mGlobalOrg[u] - pad*spa[u];
		x1[v] = mGlobalOrg[v] - pad*spa[v] + spa[v]*dimV;
		points->SetPoint(3,x1);
		tc[0] = 0.0;
		tc[1] = float(dimV)/dimsTexture[1];
		tcoords->SetTuple(3,tc);

		//
		// We now assign the pieces to the vtkPolyData.
		//
		output->SetPolys(polys);
		output->SetPoints(points);
		output->GetPointData()->SetTCoords(tcoords);
		polys->Delete();
		points->Delete();
		tcoords->Delete();
	}

	//
	//  Fill in our portion of the texture
	//
 	int ncomIn = scalars->GetNumberOfComponents();
    float *ptrIn = scalars->GetPointer(0);
	float *ptrOut = (float *)mTexture->GetScalarPointer();

	if(ptrOut == 0)
	{
		vtkErrorMacro("Texture data has not been allocated properly.");
		return;
	}

	int ijk[3], ijkmin[3], ijknum[3];

	ijk[Axis] = 0;

	ijkmin[u] = 0;
	ijkmin[v] = 0;
	ijknum[u] = dimU;
	ijknum[v] = dimV;

	if(mOffX < 0) ijkmin[u] -= mOffX;
	if(mOffY < 0) ijkmin[v] -= mOffY;
	if(mOffX+ijknum[u] > dimsTexture[0]) ijknum[u] = dimsTexture[0] - mOffX;
	if(mOffY+ijknum[v] > dimsTexture[1]) ijknum[v] = dimsTexture[1] - mOffY;

	vtkIdType off1 = 0, off2 = 0, off3;
	switch(Axis)
	{
	case 0:
		{
			//
			//  u = 1, v = 2;
			//
			off1 = dims[0];
			off2 = dims[0]*dims[1];
			break;
		}
	case 1:
		{
			//
			//  u = 0, v = 1;
			//
			off1 = 1;
			off2 = dims[0]*dims[1];
			break;
		}
	case 2:
		{
			//
			//  u = 0, v = 1;
			//
			off1 = 1;
			off2 = dims[0];
			break;
		}
	}

	off1 *= ncomIn;
	off2 *= ncomIn;
	off3 = off1 + off2;

	vtkIdType lIn, lOut;
	for(ijk[v]=ijkmin[v]; ijk[v]<ijknum[v]; ijk[v]++)
	{
		this->UpdateProgress((float)(ijk[v]-ijkmin[v])/(ijknum[v]-ijkmin[v]));
		if(this->GetAbortExecute()) break;

		for(ijk[u]=ijkmin[u]; ijk[u]<ijknum[u]; ijk[u]++)
		{
			lIn = ncomIn*(ijk[0]+dims[0]*(ijk[1]+(vtkIdType)dims[1]*ijk[2]));
			lOut = mOffX + ijk[u] + dimsTexture[0]*(mOffY+ijk[v]);
			if(isPointData)
			{
				ptrOut[lOut] = 0.25f*(ptrIn[lIn]+ptrIn[lIn+off1]+ptrIn[lIn+off2]+ptrIn[lIn+off3]);
			}
			else
			{
				ptrOut[lOut] = ptrIn[lIn];
			}
//			ptrOut[lOut] = pow10(-0.1*float(ijk[u]));
		}
	}
	mTexture->Modified();
}


void iOrthoTexturePlaneFilter::UpdateBoundaryConditions()
{
	//
	//  Do it only for master
	//
	if(mTexture==0 || mOffX!=0 || mOffY!=0) return;

	int dims[3], dimsTexture[3];
	vtkImageData *input = this->GetInput();

	//
	//  Do we have cell or point data?
	//
	bool isPointData;
	if(input->GetPointData()->GetScalars() != 0)
	{
		isPointData = true;
	}
	else if(input->GetCellData()->GetScalars() != 0)
	{
		isPointData = false;
	}
	else
	{
		//
		//  Must be quiet here for parallel execution
		//
		return;
	}

	input->GetDimensions(dims);

	int i, Axis = -1;
	for(i=0; i<3; i++)
	{
		if(dims[i] == 1) Axis = i;
	}
	if(Axis == -1) return;

    int u, v, dimU, dimV;
	iOrthoSlicer::GetUV(Axis,u,v);

	if(isPointData)
	{
		dimU = dims[u] - 1;
		dimV = dims[v] - 1;
	}
	else
	{
		dimU = dims[u];
		dimV = dims[v];
	}

	if(dimU<1 || dimV<1)
	{
		return;
	}

	mTexture->GetDimensions(dimsTexture);

	float *ptrOut = (float *)mTexture->GetScalarPointer();

	//
	//  Boundary conditions for proper interpolation
	//
	int i1off, i2off;
	bool per[3];
	this->GetLimits()->GetPeriodicities(per);

	if(dimU < dimsTexture[0])
	{
		i1off = per[u] ? dimU-1 : 0;
		i2off = per[u] ? 0 : dimU-1;

		for(i=0; i<dimV; i++)
		{
			ptrOut[dimsTexture[0]-1+dimsTexture[0]*i] = ptrOut[i1off+dimsTexture[0]*i];
			ptrOut[dimU+dimsTexture[0]*i] = ptrOut[i2off+dimsTexture[0]*i];
		}
	}

	if(dimV < dimsTexture[1])
	{
		i1off = per[v] ? dimV-1 : 0;
		i2off = per[v] ? 0 : dimV-1;

		for(i=0; i<dimU; i++)
		{
			ptrOut[i+dimsTexture[0]*(dimsTexture[1]-1)] = ptrOut[i+dimsTexture[0]*i1off];
			ptrOut[i+dimsTexture[0]*dimV] = ptrOut[i+dimsTexture[0]*i2off];
		}
	}

	//
	//  Corners
	//
	if(dimU<dimsTexture[0] && dimV<dimsTexture[1])
	{
		if(per[u] && per[v])
		{
			ptrOut[dimU+dimsTexture[0]*dimV] = ptrOut[0+dimsTexture[0]*0];
			ptrOut[dimU+dimsTexture[0]*(dimsTexture[1]-1)] = ptrOut[+dimsTexture[0]*(dimV-1)];
			ptrOut[dimsTexture[0]-1+dimsTexture[0]*dimV] = ptrOut[dimU-1+dimsTexture[0]*0];
			ptrOut[dimsTexture[0]-1+dimsTexture[0]*(dimsTexture[1]-1)] = ptrOut[dimU-1+dimsTexture[0]*(dimV-1)];
		}
		else
		{
			ptrOut[dimU+dimsTexture[0]*dimV] = 0.5f*(ptrOut[dimU-1+dimsTexture[0]*dimV]+ptrOut[dimU+dimsTexture[0]*(dimV-1)]);
			ptrOut[dimU+dimsTexture[0]*(dimsTexture[1]-1)] = 0.5f*(ptrOut[dimU-1+dimsTexture[0]*(dimsTexture[1]-1)]+ptrOut[dimU+dimsTexture[0]*0]);
			ptrOut[dimsTexture[0]-1+dimsTexture[0]*dimV] = 0.5f*(ptrOut[dimsTexture[0]-1+dimsTexture[0]*(dimV-1)]+ptrOut[0+dimsTexture[0]*dimV]);
			ptrOut[dimsTexture[0]-1+dimsTexture[0]*(dimsTexture[1]-1)] = 0.5f*(ptrOut[0+dimsTexture[0]*(dimsTexture[1]-1)]+ptrOut[dimsTexture[0]-1+dimsTexture[0]*0]);
		}
	}

	mTexture->Modified();
}

