/* IMSpector - Instant Messenger Transparent Proxy Service
 * http://www.imspector.org/
 * (c) Lawrence Manning <lawrence@aslak.net>, 2006
 *          
 * Released under the GPL v2. */

#include "imspector.h"

#define PLUGIN_NAME "Gadu-Gadu IMSpector protocol plugin"
#define PROTOCOL_NAME "Gadu-Gadu"
#define PROTOCOL_PORT 8074

extern "C"
{
	bool initprotocolplugin(struct protocolplugininfo &pprotocolplugininfo,
		class Options &options, bool debugmode);
	void closeprotocolplugin(void);
	int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer,
		int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress);
	int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength);
};

#pragma pack(2)

struct header
{
	uint32_t type;
	uint32_t length;
};

struct login
{
	uint32_t localid;
};


struct messageincoming
{
	uint32_t remoteid;
	uint32_t flags1;
	uint32_t flags2;
	uint32_t flags3;
};

struct messageoutgoing
{
	uint32_t remoteid;
	uint32_t flags1;
	uint32_t flags2;
};

#pragma pack()

#define GG_TYPE_LOGIN_LIBPURPLE 0x15
#define GG_TYPE_LOGIN_OFFICIAL 0x19
#define GG_TYPE_PING 0x08
#define GG_TYPE_INCOMING_MESSAGE 0x0a
#define GG_TYPE_OUTGOING_MESSAGE 0x0b

int packetcount = 0;
bool tracing = false;
bool localdebugmode = false;
std::string clientaddress = "Unknown";
std::string localid = "Unknown";
std::string remoteid = "Unknown";

bool initprotocolplugin(struct protocolplugininfo &protocolplugininfo,
	class Options &options, bool debugmode)
{
	if (options["gg_protocol"] != "on") return false;

	localdebugmode = debugmode;
	
	protocolplugininfo.pluginname = PLUGIN_NAME;
	protocolplugininfo.protocolname = PROTOCOL_NAME;
	protocolplugininfo.port = htons(PROTOCOL_PORT);
	
	if (options["gg_trace"] == "on") tracing = true;
	
	return true;
}

void closeprotocolplugin(void)
{
	return;
}

/* The main plugin function. See protocolplugin.cpp. */
int processpacket(bool outgoing, class Socket &incomingsock, char *replybuffer, 
	int *replybufferlength, std::vector<struct imevent> &imevents, std::string &clientaddress)
{
	struct header header;
	struct login login;
	struct messageincoming messageincoming;
	struct messageoutgoing messageoutgoing;
	char buffer[BUFFER_SIZE];
	char messagebuffer[BUFFER_SIZE];
	
	memset(&header, 0, sizeof(struct header));
	memset(&login, 0, sizeof(struct login));
	memset(&messageincoming, 0, sizeof(struct messageincoming));
	memset(&messageoutgoing, 0, sizeof(struct messageoutgoing));
	memset(buffer, 0, BUFFER_SIZE);
	memset(messagebuffer, 0, BUFFER_SIZE);
	
	/* Get the fix sized header. */
	/* TODO: Ensure endianness. Network packets will be in littleendian,
	 * which may not be our own endianess. */
	if (!incomingsock.recvalldata((char *) &header, sizeof(struct header))) 
		return 1;

	memcpy(replybuffer, &header, sizeof(struct header));
	*replybufferlength = sizeof(struct header);

	debugprint(localdebugmode, PROTOCOL_NAME ": Type: %08x Length: %d bytes", header.type, header.length);

	/* Get the following data assuming the header indicates there is some. */
	if (header.length && header.length < BUFFER_SIZE)
	{
		if (!incomingsock.recvalldata((char *) buffer, header.length)) 
			return 1;
		
		memcpy(replybuffer + sizeof(struct header), buffer, header.length);
		*replybufferlength += header.length;
	}
	
	struct imevent imevent;
	
	imevent.type = TYPE_NULL;
	imevent.timestamp = time(NULL);
	imevent.clientaddress = clientaddress;
	imevent.protocolname = PROTOCOL_NAME;
	imevent.outgoing = outgoing;
	imevent.filtered = false;
	imevent.messageextent.start = 0;
	imevent.messageextent.length = 0;
	
	switch (header.type)
	{
		case GG_TYPE_LOGIN_LIBPURPLE:
		case GG_TYPE_LOGIN_OFFICIAL:
			memcpy((char *) &login, buffer, sizeof(struct login));
			debugprint(localdebugmode, PROTOCOL_NAME ": Login packet. Local user: %d", login.localid);
			
			localid = stringprintf("%d", login.localid);
			break;
	
		case GG_TYPE_PING:
			debugprint(localdebugmode, PROTOCOL_NAME ": Ping!");
			break;

		case GG_TYPE_INCOMING_MESSAGE:
			memcpy((char *) &messageincoming, buffer, sizeof(struct messageincoming));
			debugprint(localdebugmode, PROTOCOL_NAME ": Incoming message packet. Remote user: %d", messageincoming.remoteid);
			debugprint(localdebugmode, PROTOCOL_NAME ": Incoming message packet. Flags 1: %08x Flags 2: %08x Flags 3: %08x",
				messageincoming.flags1, messageincoming.flags2, messageincoming.flags3);
		
			strncpy(messagebuffer, buffer + sizeof(struct messageincoming), BUFFER_SIZE - 1);
			debugprint(localdebugmode, PROTOCOL_NAME ": Incoming messagepacket. Message: [%s]", messagebuffer);

			remoteid = stringprintf("%d", messageincoming.remoteid);

			imevent.type = TYPE_MSG;
			imevent.remoteid = remoteid;
			imevent.eventdata = messagebuffer;
			
			imevent.messageextent.start = sizeof(struct header) + sizeof(struct messageincoming);
			imevent.messageextent.length = -1; /* NULL terminated. */
			break;
			
		case GG_TYPE_OUTGOING_MESSAGE:
			memcpy((char *) &messageoutgoing, buffer, sizeof(struct messageoutgoing));
			debugprint(localdebugmode, PROTOCOL_NAME ": Outgoing message packet. Remote user: %d", messageoutgoing.remoteid);
			debugprint(localdebugmode, PROTOCOL_NAME ": Outgoing message packet. Flags 1: %08x Flags 2: %08x",
				messageoutgoing.flags1, messageoutgoing.flags2);
	
			strncpy(messagebuffer, buffer + sizeof(struct messageoutgoing), BUFFER_SIZE - 1);
			debugprint(localdebugmode, PROTOCOL_NAME ": Outgoing message packet. Message: [%s]", messagebuffer);

			remoteid = stringprintf("%d", messageoutgoing.remoteid);

			imevent.type = TYPE_MSG;
			imevent.remoteid = remoteid;
			imevent.eventdata = messagebuffer;

			imevent.messageextent.start = sizeof(struct header) + sizeof(struct messageoutgoing);
			imevent.messageextent.length = -1; /* NULL terminated. */
			break;
			
		default:
			debugprint(localdebugmode, PROTOCOL_NAME ": Unknown packet type");
			break;
	}

	if (imevent.type != TYPE_NULL)
	{
		imevent.localid = localid;

		/* Not really needed as IDs are numeric, but WTF. */
		std::transform(imevent.localid.begin(), imevent.localid.end(), imevent.localid.begin(), tolower);
		std::transform(imevent.remoteid.begin(), imevent.remoteid.end(), imevent.remoteid.begin(), tolower);
		imevents.push_back(imevent);
	}

	/* Write out trace packets if enabled. */
	if (tracing) tracepacket("gg", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return 0;
}

int generatemessagepacket(struct response &response, char *replybuffer, int *replybufferlength)
{
	if (localid.empty() || remoteid.empty()) return 1;

	*replybufferlength = sizeof(struct header);

	struct header header;
	
	if (response.outgoing)
	{
		header.type = GG_TYPE_OUTGOING_MESSAGE;

		struct messageoutgoing messageoutgoing;
		
		messageoutgoing.remoteid = atol(remoteid.c_str());
		messageoutgoing.flags1 = 0;
		messageoutgoing.flags2 = 0x08;
		
		memcpy(replybuffer + *replybufferlength, &messageoutgoing, sizeof(struct messageoutgoing));
		
		*replybufferlength += sizeof(struct messageoutgoing);
	}
	else
	{
		header.type = GG_TYPE_INCOMING_MESSAGE;
	
		struct messageincoming messageincoming;
		
		messageincoming.remoteid = atol(remoteid.c_str());
		messageincoming.flags1 = 0;
		messageincoming.flags2 = 0;
		messageincoming.flags3 = 0x08;
		
		memcpy(replybuffer + *replybufferlength, &messageincoming, sizeof(struct messageincoming));
		
		*replybufferlength += sizeof(struct messageincoming);
	}	
	
	strncpy(&replybuffer[*replybufferlength], response.text.c_str(),
		BUFFER_SIZE - *replybufferlength - 1);
		
	*replybufferlength += response.text.length() + 1;
	
	if (*replybufferlength > BUFFER_SIZE - 1)
		*replybufferlength = BUFFER_SIZE - 1;
	
	header.length = *replybufferlength - sizeof(struct header);
	
	memcpy(replybuffer, &header, sizeof(struct header));
	
	if (tracing) tracepacket("gg-out", packetcount, replybuffer, *replybufferlength);
	
	packetcount++;
	
	return 0;
}
