/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.php.dbgp.breakpoints;

import java.beans.PropertyChangeEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.netbeans.api.debugger.Breakpoint;
import org.netbeans.api.debugger.DebuggerManager;
import org.netbeans.api.debugger.DebuggerManagerAdapter;
import org.netbeans.api.debugger.Properties;

/**
 * Listens on DebuggerManager and: - loads all breakpoints & watches on startup
 * - listens on all changes of breakpoints and watches (like breakoint / watch
 * added / removed, or some property change) and saves a new values.
 *
 * @author ads
 */
public class PersistenceManager extends DebuggerManagerAdapter {
    public static final Logger LOGGER = Logger.getLogger(PersistenceManager.class.getName());
    private static final String DEBUGGER = "debugger"; // NOI18N
    private static final String PHP_DBGP = "PHP-DBGP"; // NOI18N

    @Override
    public Breakpoint[] initBreakpoints() {
        Properties p = Properties.getDefault().getProperties(DEBUGGER).getProperties(DebuggerManager.PROP_BREAKPOINTS);
        Breakpoint[] breakpoints = (Breakpoint[]) p.getArray(PHP_DBGP, new Breakpoint[0]);
        List<Breakpoint> validBreakpoints = new ArrayList<>();
        for (Breakpoint breakpoint : breakpoints) {
            if (breakpoint != null) {
                breakpoint.addPropertyChangeListener(this);
                validBreakpoints.add(breakpoint);
            } else {
                LOGGER.warning("null stored in the array obtained from \"" + PHP_DBGP + "\" property"); // TODO: why?
            }
        }
        return validBreakpoints.toArray(new Breakpoint[validBreakpoints.size()]);
    }

    @Override
    public String[] getProperties() {
        return new String[]{
            DebuggerManager.PROP_BREAKPOINTS_INIT,
            DebuggerManager.PROP_BREAKPOINTS,
        };
    }

    @Override
    public void breakpointAdded(Breakpoint breakpoint) {
        Properties properties = Properties.getDefault().getProperties(DEBUGGER).getProperties(DebuggerManager.PROP_BREAKPOINTS);
        properties.setArray(PHP_DBGP, getBreakpoints());
        breakpoint.addPropertyChangeListener(this);
    }

    @Override
    public void breakpointRemoved(Breakpoint breakpoint) {
        Properties properties = Properties.getDefault().getProperties(DEBUGGER).getProperties(DebuggerManager.PROP_BREAKPOINTS);
        properties.setArray(PHP_DBGP, getBreakpoints());
        breakpoint.removePropertyChangeListener(this);
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        /*
         * Breakpoint could be disabled/enabled.
         * This notification are got in the case changing this property.
         */
        if (evt.getSource() instanceof Breakpoint) {
            Properties.getDefault().getProperties(DEBUGGER).getProperties(DebuggerManager.PROP_BREAKPOINTS).setArray(PHP_DBGP, getBreakpoints());
        }
    }

    private Breakpoint[] getBreakpoints() {
        Breakpoint[] bpoints = DebuggerManager.getDebuggerManager().getBreakpoints();
        List<Breakpoint> result = new ArrayList<>();
        for (Breakpoint breakpoint : bpoints) {
            // Don't store hidden breakpoints
            if (breakpoint instanceof AbstractBreakpoint) {
                result.add(breakpoint);
            }
        }
        return result.toArray(new Breakpoint[result.size()]);
    }

}
