/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <libwpd/libwpd.h>
#include <libwpg/libwpg.h>
#include <libodfgen/libodfgen.hxx>

#include "OutputFileHelper.hxx"
#include "StringDocumentHandler.hxx"

using namespace libwpd;

class OdtOutputFileHelper : public OutputFileHelper
{
public:
	OdtOutputFileHelper(const char *outFileName,const char *password) :
		OutputFileHelper(outFileName, password) {};
	~OdtOutputFileHelper() {};

	bool convertDocument(librevenge::RVNGInputStream &input, const char *password, bool isFlat)
	{
		OdtGenerator collector;
		collector.registerEmbeddedObjectHandler("image/x-wpg", &handleEmbeddedWPGObject);
		collector.registerEmbeddedImageHandler("image/x-wpg", &handleEmbeddedWPGImage);
		StringDocumentHandler stylesHandler, contentHandler, manifestHandler, metaHandler;
		if (isFlat)
			collector.addDocumentHandler(&contentHandler, ODF_FLAT_XML);
		else
		{
			collector.addDocumentHandler(&contentHandler, ODF_CONTENT_XML);
			collector.addDocumentHandler(&manifestHandler, ODF_MANIFEST_XML);
			collector.addDocumentHandler(&metaHandler, ODF_META_XML);
			collector.addDocumentHandler(&stylesHandler, ODF_STYLES_XML);
		}
		if (WPD_OK != WPDocument::parse(&input, &collector, password))
			return false;
		if (isFlat)
		{
			printf("%s\n", contentHandler.cstr());
			return true;
		}

		static const char s_mimetypeStr[] = "application/vnd.oasis.opendocument.text";
		if (!writeChildFile("mimetype", s_mimetypeStr, (char)0) ||
		        !writeChildFile("META-INF/manifest.xml", manifestHandler.cstr()) ||
		        !writeChildFile("content.xml", contentHandler.cstr()) ||
		        !writeChildFile("meta.xml", metaHandler.cstr()) ||
		        !writeChildFile("styles.xml", stylesHandler.cstr()))
			return false;

		librevenge::RVNGStringVector objects=collector.getObjectNames();
		for (unsigned i=0; i<objects.size(); ++i)
		{
			StringDocumentHandler objectHandler;
			if (collector.getObjectContent(objects[i], &objectHandler))
				writeChildFile(objects[i].cstr(), objectHandler.cstr());
		}
		return true;
	}


	bool isSupportedFormat(librevenge::RVNGInputStream &input, const char *password)
	{
		WPDConfidence confidence = WPDocument::isFileFormatSupported(&input);
		if (WPD_CONFIDENCE_EXCELLENT != confidence && WPD_CONFIDENCE_SUPPORTED_ENCRYPTION != confidence)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid WordPerfect document.\n");
			return false;
		}
		if (WPD_CONFIDENCE_SUPPORTED_ENCRYPTION == confidence && !password)
		{
			fprintf(stderr, "ERROR: The WordPerfect document is encrypted and you did not give us a password.\n");
			return false;
		}
		if (confidence == WPD_CONFIDENCE_SUPPORTED_ENCRYPTION && password &&
		        (WPD_PASSWORD_MATCH_OK != WPDocument::verifyPassword(&input, password)))
		{
			fprintf(stderr, "ERROR: The WordPerfect document is encrypted and we either\n");
			fprintf(stderr, "ERROR: don't know how to decrypt it or the given password is wrong.\n");
			return false;
		}

		return true;
	}

private:

	static bool handleEmbeddedWPGObject(const librevenge::RVNGBinaryData &data, OdfDocumentHandler *pHandler,  const OdfStreamType streamType)
	{
		OdgGenerator exporter;
		exporter.addDocumentHandler(pHandler, streamType);

		libwpg::WPGFileFormat fileFormat = libwpg::WPG_AUTODETECT;

		if (!libwpg::WPGraphics::isSupported(data.getDataStream()))
			fileFormat = libwpg::WPG_WPG1;

		return libwpg::WPGraphics::parse(data.getDataStream(), &exporter, fileFormat);
	}

	static bool handleEmbeddedWPGImage(const librevenge::RVNGBinaryData &input, librevenge::RVNGBinaryData &output)
	{
		libwpg::WPGFileFormat fileFormat = libwpg::WPG_AUTODETECT;

		if (!libwpg::WPGraphics::isSupported(input.getDataStream()))
			fileFormat = libwpg::WPG_WPG1;

		librevenge::RVNGStringVector svgOutput;
		librevenge::RVNGSVGDrawingGenerator generator(svgOutput, "");
		bool result = libwpg::WPGraphics::parse(input.getDataStream(), &generator, fileFormat);
		if (!result || svgOutput.empty() || svgOutput[0].empty())
			return false;

		output.clear();
		const char *svgHeader = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n"
		                        "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\""
		                        " \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n";
		output.append((unsigned char *)svgHeader, strlen(svgHeader));
		output.append((unsigned char *)svgOutput[0].cstr(), strlen(svgOutput[0].cstr()));

		return true;
	}
};


int printUsage(char *name)
{
	fprintf(stderr, "USAGE : %s [--stdout] --password <password> <infile> [outfile]\n", name);
	fprintf(stderr, "USAGE : Where <infile> is the WordPerfect source document\n");
	fprintf(stderr, "USAGE : and [outfile] is the odt target document. Alternately,\n");
	fprintf(stderr, "USAGE : pass '--stdout' or simply omit the [outfile] to pipe the\n");
	fprintf(stderr, "USAGE : resultant document as flat XML to standard output\n");
	fprintf(stderr, "USAGE : pass '--password <password>' to try to decrypt password\n");
	fprintf(stderr, "USAGE : protected documents.\n");
	fprintf(stderr, "USAGE : \n");
	return 1;
}


int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(argv[0]);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;
	char *password = 0;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--password"))
		{
			if (i < argc - 1)
				password = argv[++i];
		}
		else if (!strncmp(argv[i], "--password=", 11))
			password = &argv[i][11];
		else if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(argv[0]);
	}

	if (!szInputFile)
		return printUsage(argv[0]);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	OdtOutputFileHelper helper(szOutFile, 0);
	librevenge::RVNGFileStream input(szInputFile);
	if (!helper.isSupportedFormat(input, password))
		return 1;

	if (!helper.convertDocument(input, password, szOutFile==0))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
