//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Fit/Option/OptionContainer.h
//! @brief     Declares class OptionContainer.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifdef SWIG
#error no need to expose this header to Swig
#endif // SWIG
#ifndef BORNAGAIN_FIT_OPTION_OPTIONCONTAINER_H
#define BORNAGAIN_FIT_OPTION_OPTIONCONTAINER_H

#include "Base/Type/OwningVector.h"
#include "Fit/Option/MultiOption.h"
#include <map>
#include <memory>
#include <stdexcept>

//! Stores multi option (int,double,string) in a container.

class OptionContainer {
public:
    using option_t = MultiOption*;
    using container_t = OwningVector<MultiOption>;
    using iterator = container_t::Iterator;
    using const_iterator = container_t::ConstIterator;

    OptionContainer() = default;
    OptionContainer(const OptionContainer& other);
    OptionContainer& operator=(const OptionContainer& other);

    template <typename T>
    option_t addOption(const std::string& optionName, T value, const std::string& description = "");

    option_t option(const std::string& optionName);
    option_t option(const std::string& optionName) const;

    template <typename T> T optionValue(const std::string& optionName) const;

    //! Sets the value of option. Option should hold same value type already.
    template <typename T> void setOptionValue(const std::string& optionName, T value);

    iterator begin() { return m_options.begin(); }
    const_iterator begin() const { return m_options.begin(); }

    iterator end() { return m_options.end(); }
    const_iterator end() const { return m_options.end(); }

    size_t size() const { return m_options.size(); }
    bool empty() const { return size() == 0; }

protected:
    bool exists(const std::string& name);
    void swapContent(OptionContainer& other);
    container_t m_options;
};

template <typename T>
OptionContainer::option_t OptionContainer::addOption(const std::string& optionName, T value,
                                                     const std::string& description)
{
    if (exists(optionName))
        throw std::runtime_error("OptionContainer::addOption -> Error. Option '" + optionName
                                 + "' exists.");

    option_t result(new MultiOption(optionName, value, description));
    m_options.push_back(result);
    return result;
}

template <typename T> T OptionContainer::optionValue(const std::string& optionName) const
{
    return option(optionName)->get<T>();
}

template <typename T> void OptionContainer::setOptionValue(const std::string& optionName, T value)
{
    option(optionName)->value() = value;
    if (option(optionName)->value().index() != option(optionName)->defaultValue().index())
        throw std::runtime_error(
            "OptionContainer::setOptionValue -> Error. Attempt to set different"
            "type to option '"
            + optionName + "'");
}

#endif // BORNAGAIN_FIT_OPTION_OPTIONCONTAINER_H
