package org.apache.maven.doxia.book;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.maven.doxia.book.model.BookModel;

import java.io.File;
import java.util.List;
import java.util.Locale;

/**
 * An interface to create books in different output formats from a book descriptor.
 *
 * @author <a href="mailto:trygvis@inamo.no">Trygve Laugst&oslash;l</a>
 * @version $Id: BookDoxia.java 748141 2009-02-26 13:29:10Z vsiveton $
 */
public interface BookDoxia
{
    /** The plexus lookup role. */
    String ROLE = BookDoxia.class.getName();

    /**
     * Load a BookModel from a descriptor file.
     *
     * @param bookDescriptor the book descriptor file.
     * @return BookModel
     * @throws org.apache.maven.doxia.book.BookDoxiaException if the model cannot be loaded.
     */
    BookModel loadBook( File bookDescriptor )
        throws BookDoxiaException;

    /**
     * Creates a book from a BookModel using the {@link Locale#getDefault()} and UTF-8 as default encoding.
     *
     * @param book the BookModel.
     * @param bookRendererId the id of the output format.
     * @param files a list of source files.
     * @param outputDirectory the output directory.
     * @throws org.apache.maven.doxia.book.BookDoxiaException if the model cannot be loaded.
     * @see #renderBook(BookModel, String, List, File, Locale, String, String)
     * @see Locale#getDefault()
     */
    void renderBook( BookModel book, String bookRendererId, List files, File outputDirectory )
        throws BookDoxiaException;

    /**
     * Creates a book from a BookModel.
     *
     * @param book the BookModel.
     * @param bookRendererId the id of the output format.
     * @param files a list of source files.
     * @param outputDirectory the output directory.
     * @param locale the wanted locale.
     * @param inputEncoding the input encoding when processing <code>files</code>.
     * @param outputEncoding the output encoding when writing files in <code>ouputDirectory</code>.
     * @throws org.apache.maven.doxia.book.BookDoxiaException if the model cannot be loaded.
     * @since 1.1
     */
    void renderBook( BookModel book, String bookRendererId, List files, File outputDirectory, Locale locale,
                     String inputEncoding, String outputEncoding )
        throws BookDoxiaException;
}
