# Copyright (c) 2016 RIPE NCC
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from ripe.atlas.sagan import Result, ResultError
from ripe.atlas.sagan.ssl import SslResult


def test_ssl_4480():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIFBTCCAu2gAwIBAgIDDLHHMA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTEzMDEwNjE0MDA1NVoXDTEzMDcwNTE0MDA1NVowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOB9jCB8zAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMDUGA1UdEQQuMCyCDSoucHJl\\ndGljYWwuZWWgGwYIKwYBBQUHCAWgDwwNKi5wcmV0aWNhbC5lZTANBgkqhkiG9w0B\\nAQUFAAOCAgEAycddS/c47eo0WVrFxpvCIJdfvn7CYdTPpXNSg0kjapkSjYuAkcmq\\nsrScUUGMBe6tfkmkdPTuNKwRVYNJ1Wi9EYaMvJ3CVw6x9O5mgktmu0ogbIXsivwI\\nTSzGDMWcb9Of85e/ALWpK0cWIugtWO0d6v3qMWfxlYfAaYu49pttOJQOjbXAAhfR\\njE5VOcDaIlWChG48jLAyCLsMwHlyLw8D5Myb9MfTs1XxgLESO9ZTSqGEqJw+BwTJ\\nstHk/oCHo9FL/Xv5MmFcNaTpqbB60duYJ+DLLX1QiRRfLJ18G7wEiEAm6H9egupQ\\nL9MhQQLJ4o60xTrCnpqGTXTSR16jiTm70bDB0+SU3xTpNwCzdigH6ceKbPIr0cO6\\no0ump598e2JNCPsXIc+XcbLDDFgYrlnl3YnK3J+K3LC7SWPMsYdDfe+Im880tNuW\\nOlnOCDpP8408KqCp4xm0DMznmThUM34/Ia+o8Q3NWNBfuaOsJ9aA+FmgobJhih9e\\nUr9x3ByRQXcW5Cs/AMtCikKWVPsx+IA5eeyt+1i+dKBWksO40B3ADsq1O5DRYYRa\\n+dwqdX/jduqZjbyHuFH04q28j4zVDviUBQEa9UQoDM3c82dILDjbYtZ+T28sPMTa\\nbMZdcMur9E+ovrS58lIKGCvDEPSUDXHzr0tpb4A13TTnxW6pclqUyJk=\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"https","from":"77.95.64.18","fw":4480,"method":"SSL","msm_id":1006864,"prb_id":517,"src_addr":"77.95.64.18","timestamp":1362454627,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4480)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "77.95.64.18")
    assert(result.port == 443)
    assert(result.origin == "77.95.64.18")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time is None)
    assert(result.time_to_connect is None)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 517)
    assert(result.created.isoformat() == "2013-03-05T03:37:07+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2013-01-06T14:00:55+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2013-07-05T14:00:55+00:00")
    assert(result.certificates[0].checksum == "AC:D1:6F:FB:3A:F2:06:C9:92:A4:4D:2A:A2:0B:1C:37:35:25:75:A7:5D:4A:62:21:03:AD:E6:46:42:E5:65:4C")
    assert(result.certificates[0].checksum_md5 == "91:EA:DF:FF:57:6F:7C:91:F1:83:2A:9F:7F:A2:FE:E4")
    assert(result.certificates[0].checksum_sha1 == "11:00:27:07:F7:E8:0C:6B:5F:B6:A3:5A:20:19:BC:CA:22:85:E5:3C")
    assert(result.certificates[0].checksum_sha256 == "AC:D1:6F:FB:3A:F2:06:C9:92:A4:4D:2A:A2:0B:1C:37:35:25:75:A7:5D:4A:62:21:03:AD:E6:46:42:E5:65:4C")


def test_ssl_4500():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIFBTCCAu2gAwIBAgIDDLHHMA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTEzMDEwNjE0MDA1NVoXDTEzMDcwNTE0MDA1NVowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOB9jCB8zAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMDUGA1UdEQQuMCyCDSoucHJl\\ndGljYWwuZWWgGwYIKwYBBQUHCAWgDwwNKi5wcmV0aWNhbC5lZTANBgkqhkiG9w0B\\nAQUFAAOCAgEAycddS/c47eo0WVrFxpvCIJdfvn7CYdTPpXNSg0kjapkSjYuAkcmq\\nsrScUUGMBe6tfkmkdPTuNKwRVYNJ1Wi9EYaMvJ3CVw6x9O5mgktmu0ogbIXsivwI\\nTSzGDMWcb9Of85e/ALWpK0cWIugtWO0d6v3qMWfxlYfAaYu49pttOJQOjbXAAhfR\\njE5VOcDaIlWChG48jLAyCLsMwHlyLw8D5Myb9MfTs1XxgLESO9ZTSqGEqJw+BwTJ\\nstHk/oCHo9FL/Xv5MmFcNaTpqbB60duYJ+DLLX1QiRRfLJ18G7wEiEAm6H9egupQ\\nL9MhQQLJ4o60xTrCnpqGTXTSR16jiTm70bDB0+SU3xTpNwCzdigH6ceKbPIr0cO6\\no0ump598e2JNCPsXIc+XcbLDDFgYrlnl3YnK3J+K3LC7SWPMsYdDfe+Im880tNuW\\nOlnOCDpP8408KqCp4xm0DMznmThUM34/Ia+o8Q3NWNBfuaOsJ9aA+FmgobJhih9e\\nUr9x3ByRQXcW5Cs/AMtCikKWVPsx+IA5eeyt+1i+dKBWksO40B3ADsq1O5DRYYRa\\n+dwqdX/jduqZjbyHuFH04q28j4zVDviUBQEa9UQoDM3c82dILDjbYtZ+T28sPMTa\\nbMZdcMur9E+ovrS58lIKGCvDEPSUDXHzr0tpb4A13TTnxW6pclqUyJk=\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"https","from":"77.95.64.18","fw":4500,"method":"SSL","msm_id":1006864,"prb_id":517,"src_addr":"77.95.64.18","timestamp":1365577623,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4500)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "77.95.64.18")
    assert(result.port == 443)
    assert(result.origin == "77.95.64.18")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time is None)
    assert(result.time_to_connect is None)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 517)
    assert(result.created.isoformat() == "2013-04-10T07:07:03+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2013-01-06T14:00:55+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2013-07-05T14:00:55+00:00")
    assert(result.certificates[0].checksum == "AC:D1:6F:FB:3A:F2:06:C9:92:A4:4D:2A:A2:0B:1C:37:35:25:75:A7:5D:4A:62:21:03:AD:E6:46:42:E5:65:4C")
    assert(result.certificates[0].checksum_md5 == "91:EA:DF:FF:57:6F:7C:91:F1:83:2A:9F:7F:A2:FE:E4")
    assert(result.certificates[0].checksum_sha1 == "11:00:27:07:F7:E8:0C:6B:5F:B6:A3:5A:20:19:BC:CA:22:85:E5:3C")
    assert(result.certificates[0].checksum_sha256 == "AC:D1:6F:FB:3A:F2:06:C9:92:A4:4D:2A:A2:0B:1C:37:35:25:75:A7:5D:4A:62:21:03:AD:E6:46:42:E5:65:4C")


def test_ssl_4520():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIEzjCCAragAwIBAgIDDXW/MA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTEzMDcwNjA1NTQ1NFoXDTE0MDEwMjA1NTQ1NFowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOBvzCBvDAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMA0GCSqGSIb3DQEBBQUAA4IC\\nAQA3/H0E30Ukgu+icBzxOIduVai4GMrNEved6P4toz7ZjzRXv4OczpIB8KSPS9S2\\nv3EiL/YSymH9a+GrfuyZ83WqL8xNUEbCy8bkMimR0zL03dmQIM+9lCjofk80dl2l\\nqhcvjk7H6nWyVJKxVN61+XiGCznhvqZJyAZLOFNaXqyryOLmnzPMM/a6kFOijU/m\\nnhXrz56DVSYYbJVaTkJMFwNSQST+bJXz2OCWO3ovMd6iP13/w3/XL/L+p6p1wFOx\\nYPa3j6pz0zeFYH4U4xlg95MtuMwMDxJtJBPavBotpKFHT89C2Da/mxgiJalmpcnh\\nxFFRrbvkIc3oL2rW/+ZDhQpHbvMAXIhBnrBdTduSbargHaCm24Ga/R/JIxZJHP65\\n2m17gtBbqijy+4fNxQE67+cWNS3k56LyXzW2Cm1Mvz9p8IjZFalO6zf45n55f5xE\\nkK/kM2zZ5ZXJ28Y2wZ/nGABAqvQsbcd6/XtoTwcxt+GUQoCQ3AT2RR9Ar5p69yJG\\n9aMR6bjnmtTnhCigtOj7L/IpKV1b1/+nK5+00wJEF6mIvtMCMr+7zf62LTLRgM8U\\njzPNkFt0hzq3coOKOeho6pGvE9NyV+8Ed7K9H3DpYf4y5BDigOGfLVU9bnYaJdWn\\ngRkq8Xmcm73lglN49oBXrxZrxQh0LAHWgbNxuQ3iMAvmuw==\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"https","from":"77.95.64.18","fw":4520,"method":"SSL","msm_id":1006864,"msm_name":"SSLCert","prb_id":517,"src_addr":"77.95.64.18","timestamp":1378499876,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4520)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "77.95.64.18")
    assert(result.port == 443)
    assert(result.origin == "77.95.64.18")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time is None)
    assert(result.time_to_connect is None)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 517)
    assert(result.created.isoformat() == "2013-09-06T20:37:56+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2013-07-06T05:54:54+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2014-01-02T05:54:54+00:00")
    assert(result.certificates[0].checksum == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")
    assert(result.certificates[0].checksum_md5 == "38:07:D1:3C:0C:79:01:5E:F6:0E:6B:48:B4:3C:F4:59")
    assert(result.certificates[0].checksum_sha1 == "C5:31:2B:AC:B2:4F:D2:F8:47:57:2A:4F:04:D7:F9:18:91:20:C7:1F")
    assert(result.certificates[0].checksum_sha256 == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")


def test_ssl_4540():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIEzjCCAragAwIBAgIDDXW/MA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTEzMDcwNjA1NTQ1NFoXDTE0MDEwMjA1NTQ1NFowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOBvzCBvDAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMA0GCSqGSIb3DQEBBQUAA4IC\\nAQA3/H0E30Ukgu+icBzxOIduVai4GMrNEved6P4toz7ZjzRXv4OczpIB8KSPS9S2\\nv3EiL/YSymH9a+GrfuyZ83WqL8xNUEbCy8bkMimR0zL03dmQIM+9lCjofk80dl2l\\nqhcvjk7H6nWyVJKxVN61+XiGCznhvqZJyAZLOFNaXqyryOLmnzPMM/a6kFOijU/m\\nnhXrz56DVSYYbJVaTkJMFwNSQST+bJXz2OCWO3ovMd6iP13/w3/XL/L+p6p1wFOx\\nYPa3j6pz0zeFYH4U4xlg95MtuMwMDxJtJBPavBotpKFHT89C2Da/mxgiJalmpcnh\\nxFFRrbvkIc3oL2rW/+ZDhQpHbvMAXIhBnrBdTduSbargHaCm24Ga/R/JIxZJHP65\\n2m17gtBbqijy+4fNxQE67+cWNS3k56LyXzW2Cm1Mvz9p8IjZFalO6zf45n55f5xE\\nkK/kM2zZ5ZXJ28Y2wZ/nGABAqvQsbcd6/XtoTwcxt+GUQoCQ3AT2RR9Ar5p69yJG\\n9aMR6bjnmtTnhCigtOj7L/IpKV1b1/+nK5+00wJEF6mIvtMCMr+7zf62LTLRgM8U\\njzPNkFt0hzq3coOKOeho6pGvE9NyV+8Ed7K9H3DpYf4y5BDigOGfLVU9bnYaJdWn\\ngRkq8Xmcm73lglN49oBXrxZrxQh0LAHWgbNxuQ3iMAvmuw==\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"443","from":"82.161.40.58","fw":4540,"method":"SSL","msm_id":1006864,"msm_name":"SSLCert","prb_id":12,"rt":119.526,"src_addr":"192.168.178.250","timestamp":1377694418,"ttc":57.616999999999997,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4540)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "192.168.178.250")
    assert(result.port == 443)
    assert(result.origin == "82.161.40.58")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time == 119.526)
    assert(result.time_to_connect == 57.617)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 12)
    assert(result.created.isoformat() == "2013-08-28T12:53:38+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2013-07-06T05:54:54+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2014-01-02T05:54:54+00:00")
    assert(result.certificates[0].checksum == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")
    assert(result.certificates[0].checksum_md5 == "38:07:D1:3C:0C:79:01:5E:F6:0E:6B:48:B4:3C:F4:59")
    assert(result.certificates[0].checksum_sha1 == "C5:31:2B:AC:B2:4F:D2:F8:47:57:2A:4F:04:D7:F9:18:91:20:C7:1F")
    assert(result.certificates[0].checksum_sha256 == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")


def test_ssl_4550():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIEzjCCAragAwIBAgIDDXW/MA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTEzMDcwNjA1NTQ1NFoXDTE0MDEwMjA1NTQ1NFowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOBvzCBvDAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMA0GCSqGSIb3DQEBBQUAA4IC\\nAQA3/H0E30Ukgu+icBzxOIduVai4GMrNEved6P4toz7ZjzRXv4OczpIB8KSPS9S2\\nv3EiL/YSymH9a+GrfuyZ83WqL8xNUEbCy8bkMimR0zL03dmQIM+9lCjofk80dl2l\\nqhcvjk7H6nWyVJKxVN61+XiGCznhvqZJyAZLOFNaXqyryOLmnzPMM/a6kFOijU/m\\nnhXrz56DVSYYbJVaTkJMFwNSQST+bJXz2OCWO3ovMd6iP13/w3/XL/L+p6p1wFOx\\nYPa3j6pz0zeFYH4U4xlg95MtuMwMDxJtJBPavBotpKFHT89C2Da/mxgiJalmpcnh\\nxFFRrbvkIc3oL2rW/+ZDhQpHbvMAXIhBnrBdTduSbargHaCm24Ga/R/JIxZJHP65\\n2m17gtBbqijy+4fNxQE67+cWNS3k56LyXzW2Cm1Mvz9p8IjZFalO6zf45n55f5xE\\nkK/kM2zZ5ZXJ28Y2wZ/nGABAqvQsbcd6/XtoTwcxt+GUQoCQ3AT2RR9Ar5p69yJG\\n9aMR6bjnmtTnhCigtOj7L/IpKV1b1/+nK5+00wJEF6mIvtMCMr+7zf62LTLRgM8U\\njzPNkFt0hzq3coOKOeho6pGvE9NyV+8Ed7K9H3DpYf4y5BDigOGfLVU9bnYaJdWn\\ngRkq8Xmcm73lglN49oBXrxZrxQh0LAHWgbNxuQ3iMAvmuw==\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"443","from":"118.209.21.239","fw":4550,"method":"SSL","msm_id":1006864,"msm_name":"SSLCert","prb_id":33,"rt":703.38599999999997,"src_addr":"192.168.178.29","timestamp":1378042943,"ttc":349.20499999999998,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4550)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "192.168.178.29")
    assert(result.port == 443)
    assert(result.origin == "118.209.21.239")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time == 703.386)
    assert(result.time_to_connect == 349.205)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 33)
    assert(result.created.isoformat() == "2013-09-01T13:42:23+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2013-07-06T05:54:54+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2014-01-02T05:54:54+00:00")
    assert(result.certificates[0].checksum == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")
    assert(result.certificates[0].checksum_md5 == "38:07:D1:3C:0C:79:01:5E:F6:0E:6B:48:B4:3C:F4:59")
    assert(result.certificates[0].checksum_sha1 == "C5:31:2B:AC:B2:4F:D2:F8:47:57:2A:4F:04:D7:F9:18:91:20:C7:1F")
    assert(result.certificates[0].checksum_sha256 == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")


def test_ssl_4550_error():
    result_string = '{"dnserr":"non-recoverable failure in name resolution","dst_name":"pretical.ee","dst_port":"443","from":"77.95.64.18","fw":4550,"msm_id":1006864,"msm_name":"SSLCert","prb_id":517,"timestamp":1380791230,"type":"sslcert"}'
    result = Result.get(result_string)
    assert(isinstance(result, SslResult))
    assert(result.af is None)
    assert(result.firmware == 4550)
    assert(result.destination_address is None)
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address is None)
    assert(result.port == 443)
    assert(result.origin == "77.95.64.18")
    assert(result.method is None)
    assert(result.version is None)
    assert(result.response_time is None)
    assert(result.time_to_connect is None)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 517)
    assert(result.created.isoformat() == "2013-10-03T09:07:10+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 0)
    assert(result.is_error is True)
    try:
        Result.get(result_string, on_error=Result.ACTION_FAIL)
        assert False
    except ResultError:
        pass


def test_ssl_4560():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIEzjCCAragAwIBAgIDDXW/MA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTEzMDcwNjA1NTQ1NFoXDTE0MDEwMjA1NTQ1NFowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOBvzCBvDAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMA0GCSqGSIb3DQEBBQUAA4IC\\nAQA3/H0E30Ukgu+icBzxOIduVai4GMrNEved6P4toz7ZjzRXv4OczpIB8KSPS9S2\\nv3EiL/YSymH9a+GrfuyZ83WqL8xNUEbCy8bkMimR0zL03dmQIM+9lCjofk80dl2l\\nqhcvjk7H6nWyVJKxVN61+XiGCznhvqZJyAZLOFNaXqyryOLmnzPMM/a6kFOijU/m\\nnhXrz56DVSYYbJVaTkJMFwNSQST+bJXz2OCWO3ovMd6iP13/w3/XL/L+p6p1wFOx\\nYPa3j6pz0zeFYH4U4xlg95MtuMwMDxJtJBPavBotpKFHT89C2Da/mxgiJalmpcnh\\nxFFRrbvkIc3oL2rW/+ZDhQpHbvMAXIhBnrBdTduSbargHaCm24Ga/R/JIxZJHP65\\n2m17gtBbqijy+4fNxQE67+cWNS3k56LyXzW2Cm1Mvz9p8IjZFalO6zf45n55f5xE\\nkK/kM2zZ5ZXJ28Y2wZ/nGABAqvQsbcd6/XtoTwcxt+GUQoCQ3AT2RR9Ar5p69yJG\\n9aMR6bjnmtTnhCigtOj7L/IpKV1b1/+nK5+00wJEF6mIvtMCMr+7zf62LTLRgM8U\\njzPNkFt0hzq3coOKOeho6pGvE9NyV+8Ed7K9H3DpYf4y5BDigOGfLVU9bnYaJdWn\\ngRkq8Xmcm73lglN49oBXrxZrxQh0LAHWgbNxuQ3iMAvmuw==\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"443","from":"62.146.75.210","fw":4560,"method":"SSL","msm_id":1006864,"msm_name":"SSLCert","prb_id":259,"rt":71.572999999999993,"src_addr":"10.0.9.74","timestamp":1385719912,"ttc":33.630000000000003,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4560)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "10.0.9.74")
    assert(result.port == 443)
    assert(result.origin == "62.146.75.210")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time == 71.573)
    assert(result.time_to_connect == 33.63)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 259)
    assert(result.created.isoformat() == "2013-11-29T10:11:52+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2013-07-06T05:54:54+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2014-01-02T05:54:54+00:00")
    assert(result.certificates[0].checksum == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")
    assert(result.certificates[0].checksum_md5 == "38:07:D1:3C:0C:79:01:5E:F6:0E:6B:48:B4:3C:F4:59")
    assert(result.certificates[0].checksum_sha1 == "C5:31:2B:AC:B2:4F:D2:F8:47:57:2A:4F:04:D7:F9:18:91:20:C7:1F")
    assert(result.certificates[0].checksum_sha256 == "F9:C1:DC:CE:CE:C2:E8:42:00:D1:A1:B5:0D:93:09:DE:73:DA:CE:56:0A:FD:A3:BB:B1:CC:1E:2E:47:15:C3:A6")


def test_ssl_4570():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIEzjCCAragAwIBAgIDDky9MA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTE0MDEwMzIxMjI1MVoXDTE0MDcwMjIxMjI1MVowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOBvzCBvDAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMA0GCSqGSIb3DQEBBQUAA4IC\\nAQC0DDa/u4yY8e8Qaf+GVtB0mMd4EQlfbvlGAeW6+7JdEmqqZoNuzrPPYNV1RLCm\\nct7GClgDhmIWu1UWP9uXmnNK7lwYqVSykGRkbJCG1Wnwku2CqIPvRwrazmd0IAOZ\\nXO3kymgbo+pmARWIulpbxz5wzCG0K5OFSFO0Ow4p4MqdrGcid9tZ81Z+db7PbVkk\\nGAwlJya0Y8kRZS/arG+UPCPl+O2OV6ICmZ3502+nJJhZ1Ahx7stjtv69HxWoX3zQ\\n/xuBnKFIZK1EtlVKVvXcFth7akRqs0K9/h1kaxoKoPBfxpZqhjOiu6+OkdydP+CY\\nkRZsj2WDYqZlbSnzYNsqy1Asp/cfsyWHp+scUBh+f4ENDg98M9sl4tpOATH+OFD7\\nypdttWGjShYfOeQWWJF9J9Q/YFseYH1oAyQ4AV4ga4SM5BZksbsfXgEgXyxdo8DN\\nZO4PfRAtaSH+ZKrmgmrcm/CcvNn+EaW6qw7dAhJE4JNkfUAm5qMTuA35ERMkgucD\\nyu/PjjzBG2HeHrSqzb3ZoERlF8wfU0wIJeiDRtHYUfu/Hr+XA/80Uq66YrhRyqPS\\nM1x0SqeaCYdgfsmK1fB0/brDbOhNXIYbvIShcimGyjfKyHK/FL18SF676/Mc9fbe\\ngWyIey4h+S/dYCHJpVKj2HUq1Jni6n3CtxNyqF0EjR1FVA==\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"443","from":"31.24.127.132","fw":4570,"method":"SSL","msm_id":1006864,"msm_name":"SSLCert","prb_id":417,"rt":149.22300000000001,"src_addr":"31.24.127.132","timestamp":1392657784,"ttc":72.516000000000005,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4570)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "31.24.127.132")
    assert(result.port == 443)
    assert(result.origin == "31.24.127.132")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time == 149.223)
    assert(result.time_to_connect == 72.516)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 417)
    assert(result.created.isoformat() == "2014-02-17T17:23:04+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2014-01-03T21:22:51+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2014-07-02T21:22:51+00:00")
    assert(result.certificates[0].checksum == "28:E6:A2:51:CE:D1:E9:34:2E:69:41:44:55:43:41:88:4D:93:4E:67:02:09:48:8C:86:68:25:8F:5E:DF:96:98")
    assert(result.certificates[0].checksum_md5 == "97:A2:0C:C7:93:E1:4B:24:6D:EA:55:09:6E:21:A8:96")
    assert(result.certificates[0].checksum_sha1 == "2B:FE:E8:F8:4B:25:19:D4:9B:3B:18:A3:29:E1:6C:79:72:FF:96:02")
    assert(result.certificates[0].checksum_sha256 == "28:E6:A2:51:CE:D1:E9:34:2E:69:41:44:55:43:41:88:4D:93:4E:67:02:09:48:8C:86:68:25:8F:5E:DF:96:98")


def test_ssl_4600():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIEzjCCAragAwIBAgIDDky9MA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTE0MDEwMzIxMjI1MVoXDTE0MDcwMjIxMjI1MVowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOBvzCBvDAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMA0GCSqGSIb3DQEBBQUAA4IC\\nAQC0DDa/u4yY8e8Qaf+GVtB0mMd4EQlfbvlGAeW6+7JdEmqqZoNuzrPPYNV1RLCm\\nct7GClgDhmIWu1UWP9uXmnNK7lwYqVSykGRkbJCG1Wnwku2CqIPvRwrazmd0IAOZ\\nXO3kymgbo+pmARWIulpbxz5wzCG0K5OFSFO0Ow4p4MqdrGcid9tZ81Z+db7PbVkk\\nGAwlJya0Y8kRZS/arG+UPCPl+O2OV6ICmZ3502+nJJhZ1Ahx7stjtv69HxWoX3zQ\\n/xuBnKFIZK1EtlVKVvXcFth7akRqs0K9/h1kaxoKoPBfxpZqhjOiu6+OkdydP+CY\\nkRZsj2WDYqZlbSnzYNsqy1Asp/cfsyWHp+scUBh+f4ENDg98M9sl4tpOATH+OFD7\\nypdttWGjShYfOeQWWJF9J9Q/YFseYH1oAyQ4AV4ga4SM5BZksbsfXgEgXyxdo8DN\\nZO4PfRAtaSH+ZKrmgmrcm/CcvNn+EaW6qw7dAhJE4JNkfUAm5qMTuA35ERMkgucD\\nyu/PjjzBG2HeHrSqzb3ZoERlF8wfU0wIJeiDRtHYUfu/Hr+XA/80Uq66YrhRyqPS\\nM1x0SqeaCYdgfsmK1fB0/brDbOhNXIYbvIShcimGyjfKyHK/FL18SF676/Mc9fbe\\ngWyIey4h+S/dYCHJpVKj2HUq1Jni6n3CtxNyqF0EjR1FVA==\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"443","from":"31.24.127.132","fw":4600,"method":"SSL","msm_id":1006864,"msm_name":"SSLCert","prb_id":417,"rt":146.65700000000001,"src_addr":"31.24.127.132","timestamp":1395807798,"ttc":71.251000000000005,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4600)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "31.24.127.132")
    assert(result.port == 443)
    assert(result.origin == "31.24.127.132")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time == 146.657)
    assert(result.time_to_connect == 71.251)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 417)
    assert(result.created.isoformat() == "2014-03-26T04:23:18+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2014-01-03T21:22:51+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2014-07-02T21:22:51+00:00")
    assert(result.certificates[0].checksum == "28:E6:A2:51:CE:D1:E9:34:2E:69:41:44:55:43:41:88:4D:93:4E:67:02:09:48:8C:86:68:25:8F:5E:DF:96:98")
    assert(result.certificates[0].checksum_md5 == "97:A2:0C:C7:93:E1:4B:24:6D:EA:55:09:6E:21:A8:96")
    assert(result.certificates[0].checksum_sha1 == "2B:FE:E8:F8:4B:25:19:D4:9B:3B:18:A3:29:E1:6C:79:72:FF:96:02")
    assert(result.certificates[0].checksum_sha256 == "28:E6:A2:51:CE:D1:E9:34:2E:69:41:44:55:43:41:88:4D:93:4E:67:02:09:48:8C:86:68:25:8F:5E:DF:96:98")


def test_ssl_4610():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIEzjCCAragAwIBAgIDDky9MA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTE0MDEwMzIxMjI1MVoXDTE0MDcwMjIxMjI1MVowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOBvzCBvDAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMA0GCSqGSIb3DQEBBQUAA4IC\\nAQC0DDa/u4yY8e8Qaf+GVtB0mMd4EQlfbvlGAeW6+7JdEmqqZoNuzrPPYNV1RLCm\\nct7GClgDhmIWu1UWP9uXmnNK7lwYqVSykGRkbJCG1Wnwku2CqIPvRwrazmd0IAOZ\\nXO3kymgbo+pmARWIulpbxz5wzCG0K5OFSFO0Ow4p4MqdrGcid9tZ81Z+db7PbVkk\\nGAwlJya0Y8kRZS/arG+UPCPl+O2OV6ICmZ3502+nJJhZ1Ahx7stjtv69HxWoX3zQ\\n/xuBnKFIZK1EtlVKVvXcFth7akRqs0K9/h1kaxoKoPBfxpZqhjOiu6+OkdydP+CY\\nkRZsj2WDYqZlbSnzYNsqy1Asp/cfsyWHp+scUBh+f4ENDg98M9sl4tpOATH+OFD7\\nypdttWGjShYfOeQWWJF9J9Q/YFseYH1oAyQ4AV4ga4SM5BZksbsfXgEgXyxdo8DN\\nZO4PfRAtaSH+ZKrmgmrcm/CcvNn+EaW6qw7dAhJE4JNkfUAm5qMTuA35ERMkgucD\\nyu/PjjzBG2HeHrSqzb3ZoERlF8wfU0wIJeiDRtHYUfu/Hr+XA/80Uq66YrhRyqPS\\nM1x0SqeaCYdgfsmK1fB0/brDbOhNXIYbvIShcimGyjfKyHK/FL18SF676/Mc9fbe\\ngWyIey4h+S/dYCHJpVKj2HUq1Jni6n3CtxNyqF0EjR1FVA==\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"443","from":"125.238.215.240","fw":4610,"method":"SSL","msm_id":1006864,"msm_name":"SSLCert","prb_id":74,"rt":696.07500000000005,"src_addr":"10.0.0.2","timestamp":1396269382,"ttc":345.97899999999998,"type":"sslcert","ver":"3.0"}')
    assert(isinstance(result, SslResult))
    assert(result.af == 4)
    assert(result.firmware == 4610)
    assert(result.destination_address == "80.79.115.54")
    assert(result.destination_name == "pretical.ee")
    assert(result.source_address == "10.0.0.2")
    assert(result.port == 443)
    assert(result.origin == "125.238.215.240")
    assert(result.method == "SSL")
    assert(result.version == "3.0")
    assert(result.response_time == 696.075)
    assert(result.time_to_connect == 345.979)
    assert(result.measurement_id == 1006864)
    assert(result.probe_id == 74)
    assert(result.created.isoformat() == "2014-03-31T12:36:22+00:00")
    assert(isinstance(result.certificates, list))
    assert(len(result.certificates) == 1)
    assert(result.certificates[0])
    assert(result.certificates[0].subject_cn == "*.pretical.ee")
    assert(result.certificates[0].subject_o is None)
    assert(result.certificates[0].subject_c is None)
    assert(result.certificates[0].issuer_cn == "CA Cert Signing Authority")
    assert(result.certificates[0].issuer_o == "Root CA")
    assert(result.certificates[0].issuer_c is None)
    assert(result.certificates[0].valid_from.isoformat() == "2014-01-03T21:22:51+00:00")
    assert(result.certificates[0].valid_until.isoformat() == "2014-07-02T21:22:51+00:00")
    assert(result.certificates[0].checksum == "28:E6:A2:51:CE:D1:E9:34:2E:69:41:44:55:43:41:88:4D:93:4E:67:02:09:48:8C:86:68:25:8F:5E:DF:96:98")
    assert(result.certificates[0].checksum_md5 == "97:A2:0C:C7:93:E1:4B:24:6D:EA:55:09:6E:21:A8:96")
    assert(result.certificates[0].checksum_sha1 == "2B:FE:E8:F8:4B:25:19:D4:9B:3B:18:A3:29:E1:6C:79:72:FF:96:02")
    assert(result.certificates[0].checksum_sha256 == "28:E6:A2:51:CE:D1:E9:34:2E:69:41:44:55:43:41:88:4D:93:4E:67:02:09:48:8C:86:68:25:8F:5E:DF:96:98")


def test_ssl_lts():
    result = Result.get('{"lts":143,"rt":149.405,"from":"94.214.66.164","dst_name":"pretical.ee","af":4,"timestamp":1406561035,"msm_name":"SSLCert","fw":4650,"prb_id":7,"method":"SSL","cert":["-----BEGIN CERTIFICATE-----\\nMIIFBTCCAu2gAwIBAgIDDzuhMA0GCSqGSIb3DQEBCwUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTE0MDcwMzE5MDczMVoXDTE0MTIzMDE5MDczMVowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBANYOTtT86sCOGpdq6uQGCvC3r1f1+ydcWPmjAjskbU7zQ1ES79\/FPRLq\\nvjap7++poP1YFDkLx12\/BtgmyTXGqsgX6gL7AqQESDrYKE4dPygOhBTtL0Zolbag\\n0vJlOL9PJhbl3UVSlUjB+Aq7Q78RHnWO0ORr1QIrzicRomrPmC0ZZE10Z42BYZGL\\nu1JQnOxCLPICJJ3Y2Xgp73vlmnV3gGIYW2xdG76OIluaDXB\/ds3+65UehcnV64pw\\noMDzbOlZ7AH6zNsnGYaWozn+SCSaWSdzD7euoqeu4ydZ+WSJdlOzOem7P\/1TRhNs\\nmtLnbHCFA2D6LaKMynxCUZ0LdrKYGysCAwEAAaOB9jCB8zAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB\/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMDUGA1UdEQQuMCyCDSoucHJl\\ndGljYWwuZWWgGwYIKwYBBQUHCAWgDwwNKi5wcmV0aWNhbC5lZTANBgkqhkiG9w0B\\nAQsFAAOCAgEAJs08dE2oEsNrdTTSKlA00xxCWXLeNOlHK32AsK99ijDe2jMTikmo\\nuTP4VqgCWcT5hnq\/5LStY+d6D5NxdetVEIVfS2evrx6k0hQkpo0Qpw6PqJ0rkm12\\nH\/y2qVIVT3VJkLq1OGJvHXTTsIGpc3nRSUzYGIAI1XR506tDhd8c471I15Q8d8PU\\nkjBcbmev7ta1WSiRgbppbD9zQcNULi5oI1nUjiaE7\/O1DW8PrlkfXV0+H7BoV3ao\\nbLGjwU7G4oXTRHDHCXgXtugT2tyYOmQ\/+xJ3rjMBjrYISmrksnTSYSp6xsEDWS5R\\nn3gP8\/PQVapttauTZ5vw3U8JEoo9Sa7U5O5M5TYgx9W8S0BU\/z6CQCltds3EdqxL\\n6YfEjORQ9Y6m8leyhp8muLCCGuo0J9icw0Y7Bz8WKx6RLZ6Sjq2Ezf9S6qfnKKPy\\nV2mY9JvBXTUsOGkPfIt0jN7EspLGM4Vx4UPxtMDeCnjUNzc76Xp\/HiEPFb2Iigaa\\nvOhLYPCI\/JiY06epFg3+Dsw4TW0LYVkHmTgAATw2xxPFMlL3gcebVCuxxog2+Ih9\\nwVg5K5hMwG+R6zvr+2WDcebpdFSTscibHywyey7tJUHNHRVi+8rnWxZjaQCwKGCE\\nVp6CMVtl39lL+MKzqJaeUORvMil8ye8jmOH4q7489t8QCJY9VN1S2tA=\\n-----END CERTIFICATE-----"],"dst_port":"443","ttc":94.057,"src_addr":"10.71.1.151","group_id":1006864,"type":"sslcert","dst_addr":"80.79.115.54","ver":"3.0","msm_id":1006864}')
    assert(result.seconds_since_sync == 143)


def test_self_signed():
    ca_signed = Result.get('{"lts":663,"rt":27.807,"from":"2001:67c:2e8:13:220:4aff:fec6:cc6e","dst_name":"2a00:1450:4005:809::1010","af":6,"timestamp":1410867341,"msm_name":"SSLCert","fw":4660,"prb_id":15,"method":"SSL","cert":["-----BEGIN CERTIFICATE-----\\nMIIEdjCCA16gAwIBAgIIcPRu2ubZ1kMwDQYJKoZIhvcNAQEFBQAwSTELMAkGA1UE\\nBhMCVVMxEzARBgNVBAoTCkdvb2dsZSBJbmMxJTAjBgNVBAMTHEdvb2dsZSBJbnRl\\ncm5ldCBBdXRob3JpdHkgRzIwHhcNMTQwODI3MTE0MDU1WhcNMTQxMTI1MDAwMDAw\\nWjBoMQswCQYDVQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEWMBQGA1UEBwwN\\nTW91bnRhaW4gVmlldzETMBEGA1UECgwKR29vZ2xlIEluYzEXMBUGA1UEAwwOd3d3\\nLmdvb2dsZS5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDe2VDg\\n3Yxd0Sz7brxxcDAx4SdhOFIzkOOzL+7MJPWGbBEhGHrWUrVa\/0pxFmN8hfNaqm9N\\nd1E3R35j3uJVP2gze3V0+SayfTAKwjKbYf17aU6Y1KB4qIBoDumdqAZvBgG6Ouif\\nrUG9hMs2UjFogH6kvoyfBOVtrrZ82MVHSaxJC55RIGrbzV01r0iFDGHJUIppeOet\\nwM2R7oi5UGTwhP3zcfplrb6hMPkGE3kTkqyj5GtCcsxBT6LzGRE2ttphQDOeW6r2\\nXjUyx\/qLiLna+\/JLphG2F3MgDkdB6NvOKyHPvGisO2k\/Y6a8g3wahD\/hs5NNhNVw\\no8OvRlajtAvuPy6tAgMBAAGjggFBMIIBPTAdBgNVHSUEFjAUBggrBgEFBQcDAQYI\\nKwYBBQUHAwIwGQYDVR0RBBIwEIIOd3d3Lmdvb2dsZS5jb20waAYIKwYBBQUHAQEE\\nXDBaMCsGCCsGAQUFBzAChh9odHRwOi8vcGtpLmdvb2dsZS5jb20vR0lBRzIuY3J0\\nMCsGCCsGAQUFBzABhh9odHRwOi8vY2xpZW50czEuZ29vZ2xlLmNvbS9vY3NwMB0G\\nA1UdDgQWBBQzxbnboMDL+J0oSwvalxaNuxIt1TAMBgNVHRMBAf8EAjAAMB8GA1Ud\\nIwQYMBaAFErdBhYbvPZotXb1gba7Yhq6WoEvMBcGA1UdIAQQMA4wDAYKKwYBBAHW\\neQIFATAwBgNVHR8EKTAnMCWgI6Ahhh9odHRwOi8vcGtpLmdvb2dsZS5jb20vR0lB\\nRzIuY3JsMA0GCSqGSIb3DQEBBQUAA4IBAQAKR2+u6lbPzPElTBcb6pqyQ9TLa36v\\nBSq4ZzIV4XfvTidlav5uGpgMOUKJSgg43qN+a3pia+TXLMArxg2LZR3Ioz3zhIwJ\\nblLhHDxBOM8ZY2\/u6zFOt\/rjf+i\/DltTjG7TlIVwBxyEV5hPtsb2S2uupnsOEfM2\\nA76n\/u6OT9AcNWOzdKCBD+ifqQf4x5tyFrLOHpS7MM6jwK\/LmKGlg4kpd8qQtehU\\nPWjtBglxle0CWZXemDUdBGC1Jy9xdKMRBX1FGUzvF5KDxXScsCtNg9uXV8Bhy0bh\\nbnc6HfiBu6aXU7HPTdhl6uY39b5sA5H8WdLoQI+6FdE755AhhW5ZFG0E\\n-----END CERTIFICATE-----","-----BEGIN CERTIFICATE-----\\nMIIEBDCCAuygAwIBAgIDAjppMA0GCSqGSIb3DQEBBQUAMEIxCzAJBgNVBAYTAlVT\\nMRYwFAYDVQQKEw1HZW9UcnVzdCBJbmMuMRswGQYDVQQDExJHZW9UcnVzdCBHbG9i\\nYWwgQ0EwHhcNMTMwNDA1MTUxNTU1WhcNMTUwNDA0MTUxNTU1WjBJMQswCQYDVQQG\\nEwJVUzETMBEGA1UEChMKR29vZ2xlIEluYzElMCMGA1UEAxMcR29vZ2xlIEludGVy\\nbmV0IEF1dGhvcml0eSBHMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB\\nAJwqBHdc2FCROgajguDYUEi8iT\/xGXAaiEZ+4I\/F8YnOIe5a\/mENtzJEiaB0C1NP\\nVaTOgmKV7utZX8bhBYASxF6UP7xbSDj0U\/ck5vuR6RXEz\/RTDfRK\/J9U3n2+oGtv\\nh8DQUB8oMANA2ghzUWx\/\/zo8pzcGjr1LEQTrfSTe5vn8MXH7lNVg8y5Kr0LSy+rE\\nahqyzFPdFUuLH8gZYR\/Nnag+YyuENWllhMgZxUYi+FOVvuOAShDGKuy6lyARxzmZ\\nEASg8GF6lSWMTlJ14rbtCMoU\/M4iarNOz0YDl5cDfsCx3nuvRTPPuj5xt970JSXC\\nDTWJnZ37DhF5iR43xa+OcmkCAwEAAaOB+zCB+DAfBgNVHSMEGDAWgBTAephojYn7\\nqwVkDBF9qn1luMrMTjAdBgNVHQ4EFgQUSt0GFhu89mi1dvWBtrtiGrpagS8wEgYD\\nVR0TAQH\/BAgwBgEB\/wIBADAOBgNVHQ8BAf8EBAMCAQYwOgYDVR0fBDMwMTAvoC2g\\nK4YpaHR0cDovL2NybC5nZW90cnVzdC5jb20vY3Jscy9ndGdsb2JhbC5jcmwwPQYI\\nKwYBBQUHAQEEMTAvMC0GCCsGAQUFBzABhiFodHRwOi8vZ3RnbG9iYWwtb2NzcC5n\\nZW90cnVzdC5jb20wFwYDVR0gBBAwDjAMBgorBgEEAdZ5AgUBMA0GCSqGSIb3DQEB\\nBQUAA4IBAQA21waAESetKhSbOHezI6B1WLuxfoNCunLaHtiONgaX4PCVOzf9G0JY\\n\/iLIa704XtE7JW4S615ndkZAkNoUyHgN7ZVm2o6Gb4ChulYylYbc3GrKBIxbf\/a\/\\nzG+FA1jDaFETzf3I93k9mTXwVqO94FntT0QJo544evZG0R0SnU++0ED8Vf4GXjza\\nHFa9llF7b1cq26KqltyMdMKVvvBulRP\/F\/A8rLIQjcxz++iPAsbw+zOzlTvjwsto\\nWHPbqCRiOwY1nQ2pM714A5AuTHhdUDqB1O6gyHA43LL5Z\/qHQF1hwFGPa4NrzQU6\\nyuGnBXj8ytqU0CwIPX4WecigUCAkVDNx\\n-----END CERTIFICATE-----","-----BEGIN CERTIFICATE-----\\nMIIDfTCCAuagAwIBAgIDErvmMA0GCSqGSIb3DQEBBQUAME4xCzAJBgNVBAYTAlVT\\nMRAwDgYDVQQKEwdFcXVpZmF4MS0wKwYDVQQLEyRFcXVpZmF4IFNlY3VyZSBDZXJ0\\naWZpY2F0ZSBBdXRob3JpdHkwHhcNMDIwNTIxMDQwMDAwWhcNMTgwODIxMDQwMDAw\\nWjBCMQswCQYDVQQGEwJVUzEWMBQGA1UEChMNR2VvVHJ1c3QgSW5jLjEbMBkGA1UE\\nAxMSR2VvVHJ1c3QgR2xvYmFsIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB\\nCgKCAQEA2swYYzD99BcjGlZ+W988bDjkcbd4kdS8odhM+KhDtgPpTSEHCIjaWC9m\\nOSm9BXiLnTjoBbdqfnGk5sRgprDvgOSJKA+eJdbtg\/OtppHHmMlCGDUUna2YRpIu\\nT8rxh0PBFpVXLVDviS2Aelet8u5fa9IAjbkU+BQVNdnARqN7csiRv8lVK83Qlz6c\\nJmTM386DGXHKTubU1XupGc1V3sjs0l44U+VcT4wt\/lAjNvxm5suOpDkZALeVAjmR\\nCw7+OC7RHQWa9k0+bw8HHa8sHo9gOeL6NlMTOdReJivbPagUvTLrGAMoUgRx5asz\\nPeE4uwc2hGKceeoWMPRfwCvocWvk+QIDAQABo4HwMIHtMB8GA1UdIwQYMBaAFEjm\\naPkr0rKV10fYIyAQTzOYkJ\/UMB0GA1UdDgQWBBTAephojYn7qwVkDBF9qn1luMrM\\nTjAPBgNVHRMBAf8EBTADAQH\/MA4GA1UdDwEB\/wQEAwIBBjA6BgNVHR8EMzAxMC+g\\nLaArhilodHRwOi8vY3JsLmdlb3RydXN0LmNvbS9jcmxzL3NlY3VyZWNhLmNybDBO\\nBgNVHSAERzBFMEMGBFUdIAAwOzA5BggrBgEFBQcCARYtaHR0cHM6Ly93d3cuZ2Vv\\ndHJ1c3QuY29tL3Jlc291cmNlcy9yZXBvc2l0b3J5MA0GCSqGSIb3DQEBBQUAA4GB\\nAHbhEm5OSxYShjAGsoEIz\/AIx8dxfmbuwu3UOx\/\/8PDITtZDOLC5MH0Y0FWDomrL\\nNhGc6Ehmo21\/uBPUR\/6LWlxz\/K7ZGzIZOKuXNBSqltLroxwUCEm2u+WR74M26x1W\\nb8ravHNjkOR\/ez4iyz0H7V84dJzjA1BOoa+Y7mHyhD8S\\n-----END CERTIFICATE-----"],"dst_port":"443","ttc":11.956,"src_addr":"2001:67c:2e8:13:220:4aff:fec6:cc6e","group_id":1012449,"type":"sslcert","dst_addr":"2a00:1450:4005:809::1010","ver":"3.0","msm_id":1012449}')
    self_signed = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIDEDCCAfgCCQCKlfFLvltvwjANBgkqhkiG9w0BAQUFADBKMQswCQYDVQQGEwJI\\nVTEaMBgGA1UEAwwRa2FkYXJrYS5raXN0ZWwuaHUxHzAdBgkqhkiG9w0BCQEWEGtp\\nc3RlbEBraXN0ZWwuaHUwHhcNMTIxMDA0MDg1NTAzWhcNMjIxMDAyMDg1NTAzWjBK\\nMQswCQYDVQQGEwJIVTEaMBgGA1UEAwwRa2FkYXJrYS5raXN0ZWwuaHUxHzAdBgkq\\nhkiG9w0BCQEWEGtpc3RlbEBraXN0ZWwuaHUwggEiMA0GCSqGSIb3DQEBAQUAA4IB\\nDwAwggEKAoIBAQC7rhzio/gldiYiNjAANOgE6s1fns6BQ+yWvwXdEwm05+xwf+iD\\nP4LDzMrjsWDwyw61vdNr/HQ5nCnDvM+M3nRMX0NiOvvufm8X1js4SdrX2c+gSjkD\\n5z42gnGhgMYrcxsDPpJYwmeD3mWI/2a+BSviszWZUevMbAOMQLVLnnNR6mVUbanS\\nNXDQdT3As3yQx6J1OoLECDbuSms883dLqh2oi/C7U/DDzPBTn8rMZPzFw7SrehiM\\ntkvSn0Ge7ptRhBUYxGCObYVdakq+GU6giObsauz0FREMoef1YTlgnKHgkBjGJgQc\\nteiSFQzzP+TKfhY8qKI20AhmN4BP/TgbJcUDAgMBAAEwDQYJKoZIhvcNAQEFBQAD\\nggEBAG1szxqtff2XlRT7G5CMA5dH0o3hXUQQtRXQt3EeOzxcS0mQcLwN353uzCQp\\noOSXMF0ZD5kONZqkxZNYjwVWnyBH1d/A3bRvVNw9X038GecvzUX/VF8/PnzGYxyb\\ndPM3wCjRpaod2vpr/WwfOxWIG+Ie0u4Gzht/1UHEW1XGtOLJ8ee/SY95E/FXOTzc\\n9KtVC6lT8B+cpp3goEk4yGTdl3Uq6wnWrkK/OLg91x0/ZMK+0dIzuO1tFC5A5EH4\\ntZLxwscfsxFoV2geZGB21qKDcf1fKlBcsTIt9SiWg37rYSEmrc5hIkrQJ9mGM1+4\\nyruypEG+EAtzD9O3GrvNdgcZ0ps=\\n-----END CERTIFICATE-----"],"dst_addr":"85.10.198.82","dst_name":"85.10.198.82","dst_port":"443","from":"109.201.224.30","fw":4660,"group_id":1766133,"lts":178,"method":"SSL","msm_id":1766133,"msm_name":"SSLCert","prb_id":2965,"rt":3103.552,"src_addr":"10.119.35.124","timestamp":1410960180,"ttc":3046.154,"type":"sslcert","ver":"3.0"}')
    assert(ca_signed.is_self_signed == False)
    assert(self_signed.is_self_signed == True)


def test_non_ascii():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----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\nJsuIBXUOqcfpIJ5N39GuS834HhKcIAk81A==\n-----END CERTIFICATE-----","-----BEGIN CERTIFICATE-----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\np2wmoFf+gA5JQwKaRcbkEMlsxXaxqwLOyv7YhHbEAW0DscFfuFTsb02wGps=\n-----END CERTIFICATE-----","-----BEGIN CERTIFICATE-----\nMIIFdzCCBF+gAwIBAgIQE+oocFv07O0MNmMJgGFDNjANBgkqhkiG9w0BAQwFADBv\nMQswCQYDVQQGEwJTRTEUMBIGA1UEChMLQWRkVHJ1c3QgQUIxJjAkBgNVBAsTHUFk\nZFRydXN0IEV4dGVybmFsIFRUUCBOZXR3b3JrMSIwIAYDVQQDExlBZGRUcnVzdCBF\neHRlcm5hbCBDQSBSb290MB4XDTAwMDUzMDEwNDgzOFoXDTIwMDUzMDEwNDgzOFow\ngYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtK\nZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYD\nVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIICIjAN\nBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAgBJlFzYOw9sIs9CsVw127c0n00yt\nUINh4qogTQktZAnczomfzD2p7PbPwdzx07HWezcoEStH2jnGvDoZtF+mvX2do2NC\ntnbyqTsrkfjib9DsFiCQCT7i6HTJGLSR1GJk23+jBvGIGGqQIjy8/hPwhxR79uQf\njtTkUcYRZ0YIUcuGFFQ/vDP+fmyc/xadGL1RjjWmp2bIcmfbIWax1Jt4A8BQOujM\n8Ny8nkz+rwWWNR9XWrf/zvk9tyy29lTdyOcSOk2uTIq3XJq0tyA9yn8iNK5+O2hm\nAUTnAU5GU5szYPeUvlM3kHND8zLDU+/bqv50TmnHa4xgk97Exwzf4TKuzJM7UXiV\nZ4vuPVb+DNBpDxsP8yUmazNt925H+nND5X4OpWaxKXwyhGNVicQNwZNUMBkTrNN9\nN6frXTpsNVzbQdcS2qlJC9/YgIoJk2KOtWbPJYjNhLixP6Q5D9kCnusSTJV882sF\nqV4Wg8y4Z+LoE53MW4LTTLPtW//e5XOsIzstAL81VXQJSdhJWBp/kjbmUZIO8yZ9\nHE0XvMnsQybQv0FfQKlERPSZ51eHnlAfV1SoPv10Yy+xUGUJ5lhCLkMaTLTwJUdZ\n+gQek9QmRkpQgbLevni3/GcV4clXhB4PY9bpYrrWX1Uu6lzGKAgEJTm4Diup8kyX\nHAc/DVL17e8vgg8CAwEAAaOB9DCB8TAfBgNVHSMEGDAWgBStvZh6NLQm9/rEJlTv\nA73gJMtUGjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswDgYDVR0PAQH/\nBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wEQYDVR0gBAowCDAGBgRVHSAAMEQGA1Ud\nHwQ9MDswOaA3oDWGM2h0dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9BZGRUcnVzdEV4\ndGVybmFsQ0FSb290LmNybDA1BggrBgEFBQcBAQQpMCcwJQYIKwYBBQUHMAGGGWh0\ndHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggEBAJNl9jeD\nlQ9ew4IcH9Z35zyKwKoJ8OkLJvHgwmp1ocd5yblSYMgpEg7wrQPWCcR23+WmgZWn\nRtqCV6mVksW2jwMibDN3wXsyF24HzloUQToFJBv2FAY7qCUkDrvMKnXduXBBP3zQ\nYzYhBx9G/2CkkeFnvN4ffhkUyWNnkepnB2u0j4vAbkN9w6GAbLIevFOFfdyQoaS8\nLe9Gclc1Bb+7RrtubTeZtv8jkpHGbkD4jylW6l/VXxRTrPBPYer3IsynVgviuDQf\nJtl7GQVoP7o81DgGotPmjw7jtHFtQELFhLRAlSv0ZaBIefYdgWOWnU914Ph85I6p\n0fKtirOMxyHNwu8=\n-----END CERTIFICATE-----"],"dst_addr":"178.248.232.33","dst_name":"www.lamoda.ru","dst_port":"443","from":"84.248.94.134","fw":4700,"group_id":2198464,"lts":27,"method":"TLS","msm_id":2198464,"msm_name":"SSLCert","prb_id":20782,"rt":60.96,"src_addr":"192.168.1.191","timestamp":1438122124,"ttc":26.34,"type":"sslcert","ver":"1.2"}')
    assert(isinstance(result, SslResult))


def test_alert():
    result = Result.get('{"lts":60,"rt":65.88887,"from":"134.197.42.163","dst_name":"aus5.mozilla.org","af":4,"timestamp":1452783251,"msm_name":"SSLCert","fw":4720,"alert":{"decription":40,"level":2},"method":"TLS","prb_id":10008,"dst_port":"443","ttc":31.621485,"src_addr":"134.197.42.163","group_id":3322890,"type":"sslcert","dst_addr":"63.245.213.47","ver":"1.2","msm_id":3322890}')
    assert(result.alert is not None)
    assert(result.alert.level == 2)
    assert(result.alert.description == 40)
    assert(result.alert.description_string == "handshake_failure")
    assert(result.is_error is True)
    assert(result.error_message == "handshake_failure")
    result = Result.get('{"lts":60,"rt":65.88887,"from":"134.197.42.163","dst_name":"aus5.mozilla.org","af":4,"timestamp":1452783251,"msm_name":"SSLCert","fw":4720,"alert":{"decription":99,"level":2},"method":"TLS","prb_id":10008,"dst_port":"443","ttc":31.621485,"src_addr":"134.197.42.163","group_id":3322890,"type":"sslcert","dst_addr":"63.245.213.47","ver":"1.2","msm_id":3322890}')
    assert(result.alert is not None)
    assert(result.alert.level == 2)
    assert(result.alert.description == 99)
    assert(result.alert.description_string == "Unknown")
    assert(result.is_error is True)
    assert(result.error_message == "Unknown")
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIFBTCCAu2gAwIBAgIDDLHHMA0GCSqGSIb3DQEBBQUAMHkxEDAOBgNVBAoTB1Jv\\nb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEiMCAGA1UEAxMZ\\nQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJARYSc3VwcG9y\\ndEBjYWNlcnQub3JnMB4XDTEzMDEwNjE0MDA1NVoXDTEzMDcwNTE0MDA1NVowGDEW\\nMBQGA1UEAxQNKi5wcmV0aWNhbC5lZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC\\nAQoCggEBAMS+vX7gA8TvzFwxryFFRj1OyQjnW88GvfMuGhKJopalG1EB103oRsxi\\nMcXqwFZUicpqLKHW4lCHcRuhpKoZp8EOILnRAJRKFOjgIrcHQ02Xn4Lf/ewl601h\\n5qxqt1keU1P8j+u9m7zZN+vOoNlEKZ5SnZhysAAYqr/XIt1WY2cji/4GxjF+q1OH\\nIl5zddkIfnE52UbREKKlIakfFdj/c6GXqqsP2QTmm4x2HitCD964tZ06fA9BitQj\\nnnBXNhtm2MCuBIPBSq0/C7LREwmfnqxCFqE7iqEPNIQ2IT2D4Gh4c+nIZHqYKvCV\\nP3zh3aUaBj1o5Lo83IDdXCKAIiQRFMkCAwEAAaOB9jCB8zAMBgNVHRMBAf8EAjAA\\nMA4GA1UdDwEB/wQEAwIDqDA0BgNVHSUELTArBggrBgEFBQcDAgYIKwYBBQUHAwEG\\nCWCGSAGG+EIEAQYKKwYBBAGCNwoDAzAzBggrBgEFBQcBAQQnMCUwIwYIKwYBBQUH\\nMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5vcmcvMDEGA1UdHwQqMCgwJqAkoCKGIGh0\\ndHA6Ly9jcmwuY2FjZXJ0Lm9yZy9yZXZva2UuY3JsMDUGA1UdEQQuMCyCDSoucHJl\\ndGljYWwuZWWgGwYIKwYBBQUHCAWgDwwNKi5wcmV0aWNhbC5lZTANBgkqhkiG9w0B\\nAQUFAAOCAgEAycddS/c47eo0WVrFxpvCIJdfvn7CYdTPpXNSg0kjapkSjYuAkcmq\\nsrScUUGMBe6tfkmkdPTuNKwRVYNJ1Wi9EYaMvJ3CVw6x9O5mgktmu0ogbIXsivwI\\nTSzGDMWcb9Of85e/ALWpK0cWIugtWO0d6v3qMWfxlYfAaYu49pttOJQOjbXAAhfR\\njE5VOcDaIlWChG48jLAyCLsMwHlyLw8D5Myb9MfTs1XxgLESO9ZTSqGEqJw+BwTJ\\nstHk/oCHo9FL/Xv5MmFcNaTpqbB60duYJ+DLLX1QiRRfLJ18G7wEiEAm6H9egupQ\\nL9MhQQLJ4o60xTrCnpqGTXTSR16jiTm70bDB0+SU3xTpNwCzdigH6ceKbPIr0cO6\\no0ump598e2JNCPsXIc+XcbLDDFgYrlnl3YnK3J+K3LC7SWPMsYdDfe+Im880tNuW\\nOlnOCDpP8408KqCp4xm0DMznmThUM34/Ia+o8Q3NWNBfuaOsJ9aA+FmgobJhih9e\\nUr9x3ByRQXcW5Cs/AMtCikKWVPsx+IA5eeyt+1i+dKBWksO40B3ADsq1O5DRYYRa\\n+dwqdX/jduqZjbyHuFH04q28j4zVDviUBQEa9UQoDM3c82dILDjbYtZ+T28sPMTa\\nbMZdcMur9E+ovrS58lIKGCvDEPSUDXHzr0tpb4A13TTnxW6pclqUyJk=\\n-----END CERTIFICATE-----"],"dst_addr":"80.79.115.54","dst_name":"pretical.ee","dst_port":"https","from":"77.95.64.18","fw":4480,"method":"SSL","msm_id":1006864,"prb_id":517,"src_addr":"77.95.64.18","timestamp":1362454627,"type":"sslcert","ver":"3.0"}')
    assert(result.alert is None)
    assert(result.is_error is False)

def test_san_extension():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----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\nrum/OChG\n-----END CERTIFICATE-----"],"dst_addr":"193.0.6.158","dst_name":"atlas.ripe.net","dst_port":"443","from":"86.82.178.27","fw":4760,"lts":133,"method":"TLS","msm_id":14002,"msm_name":"SSLCert","prb_id":10951,"rt":51.558465,"src_addr":"192.168.180.22","timestamp":1490659208,"ttc":14.88238,"type":"sslcert","ver":"1.2"}')
    ext = result.certificates[0].extensions
    assert(ext and len(ext['subjectAltName'])==54)

def test_ssl_error():
    result = Result.get('{"dst_name": "example.com","dst_port": "443","error": "address not allowed","from":"10.1.1.1","fw": 4760,"group_id": 1234567,"lts": 10,"msm_id": 1234567,"msm_name": "SSLCert","prb_id": 9999,"timestamp": 1495451999,"type": "sslcert"}')
    assert(result.is_error is True)
    assert(result.error_message == "address not allowed")

def test_ssl_x509_san():
    result = Result.get('{"lts":45,"rt":67.073945,"msm_id":8844155,"from":"185.116.201.100","dst_name":"tinkoff.ru","af":4,"timestamp":1497464532,"fw":4780,"cert":["-----BEGIN CERTIFICATE-----\nMIIEsTCCA5mgAwIBAgIQV1CAInv97wvzUjqHD0BbyzANBgkqhkiG9w0BAQsFADBB\nMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMdGhhd3RlLCBJbmMuMRswGQYDVQQDExJ0\naGF3dGUgU1NMIENBIC0gRzIwHhcNMTUxMDIwMDAwMDAwWhcNMTcxMjE4MjM1OTU5\nWjBqMQswCQYDVQQGEwJSVTEPMA0GA1UECBMGTW9zY293MQ8wDQYDVQQHFAZNb3Nj\nb3cxFTATBgNVBAoUDFRpbmtvZmYgQmFuazELMAkGA1UECxQCSVQxFTATBgNVBAMU\nDCoudGlua29mZi5ydTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAN4W\ncKDTHE470BZWvUBpgce4nMWH9lNa2KWCbrcFMUcqtMaFVh4tp53gKvlvaJyecQ9y\noQBrJycdp7Du8R7HFUwgALBMxjVsrSdai3wS6txCpXJCuT+92spwyuVOT3ZGjXip\n5EwVxXuXNMbi3+alSuX1ocM3m9FWas\/fvhreQZYxpx6CH0+eqpinbTOwYc\/7gvh0\nMJdl+ESN9j9FE87aCTC\/VMakxOIRHsyApG0XNegUj3Fs2HO0XrIxMSqSrCuZqs0q\nUXINGTIoVJVgU4fzuJad2jLrzePDkliKZDZ6JZaeSKQsGgB7mNbCFwCW1oUyepHn\nASHG7BrAiP6tBbiwLHMCAwEAAaOCAXowggF2MCMGA1UdEQQcMBqCDCoudGlua29m\nZi5ydYIKdGlua29mZi5ydTAJBgNVHRMEAjAAMG4GA1UdIARnMGUwYwYGZ4EMAQIC\nMFkwJgYIKwYBBQUHAgEWGmh0dHBzOi8vd3d3LnRoYXd0ZS5jb20vY3BzMC8GCCsG\nAQUFBwICMCMMIWh0dHBzOi8vd3d3LnRoYXd0ZS5jb20vcmVwb3NpdG9yeTAOBgNV\nHQ8BAf8EBAMCBaAwHwYDVR0jBBgwFoAUwk9IV\/zRT5rAXTh9DgXb2S61UmAwKwYD\nVR0fBCQwIjAgoB6gHIYaaHR0cDovL3RqLnN5bWNiLmNvbS90ai5jcmwwHQYDVR0l\nBBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMFcGCCsGAQUFBwEBBEswSTAfBggrBgEF\nBQcwAYYTaHR0cDovL3RqLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYaaHR0cDovL3Rq\nLnN5bWNiLmNvbS90ai5jcnQwDQYJKoZIhvcNAQELBQADggEBAK1I1sCTFQn86Bhu\n78+TPM\/DNjTP1LhFOfHIYlN4askbNur1KI5i52Sqpq539Q43LWQbWZzIx1YsCsfb\nLevVZPmVGxPMgdaqs70rhAp0B1zsWBZd0ImObPTNxNYlQQwmH5S4IWa\/cxKZ9qj0\n\/vm5Wkd6eTuAfSEzOAmzDWwnTnv+aI0j3CNL4iccICMDoS0RJqoYtIzH+PGY7S21\n7xNHUlADsHa\/AeBEX6\/Ru12GMGtfSkMeYamEcrjUc\/VdX\/tIOFAMQh3iiZtpPvEQ\not8o4U20tZ7bjsRKGw2uKNfZyv9hi6flNMRHk9eS2Hi0KpOU+r462ttY8JXxUetp\nDaymEBk=\n-----END CERTIFICATE-----","-----BEGIN CERTIFICATE-----\nMIIEsjCCA5qgAwIBAgIQFofWiG3iMAaFIz2\/Eb9llzANBgkqhkiG9w0BAQsFADCB\nqTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYGA1UECxMf\nQ2VydGlmaWNhdGlvbiBTZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UECxMvKGMpIDIw\nMDYgdGhhd3RlLCBJbmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNlIG9ubHkxHzAdBgNV\nBAMTFnRoYXd0ZSBQcmltYXJ5IFJvb3QgQ0EwHhcNMTMxMDMxMDAwMDAwWhcNMjMx\nMDMwMjM1OTU5WjBBMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMdGhhd3RlLCBJbmMu\nMRswGQYDVQQDExJ0aGF3dGUgU1NMIENBIC0gRzIwggEiMA0GCSqGSIb3DQEBAQUA\nA4IBDwAwggEKAoIBAQCy\/Ab7BJPS6lkgO0SFl1I55xDweuCwlEDaRvgMKLu5zmA4\nP9LYEUIbka1J7o\/H3mzeN2\/9iyA8bed009zVJIhBgInuNr7E1b6NUxOq5KW4kwq+\n7NrNPNQyVu\/QTqC4l7s5UB5uZcP9ss7gWalICcb+vq78PjuBIJeLj0bfYGQHdbsb\nhjifR3s0zqHRl6122J+3Jtt5gDZI8sU3+NkyrnykU4HHmaFUOC9PdaC7WqW7zawC\nWxkC1RMYp86sdFUSBYubopVGZHI4zVobOhanvnGZjFQDuJZsAdM+Bpg\/IYE7An4A\nR1MBHg5GQ\/tLLdwLGugvmPh+0ZmrE2ykF95v9hX1AgMBAAGjggE7MIIBNzASBgNV\nHRMBAf8ECDAGAQH\/AgEAMA4GA1UdDwEB\/wQEAwIBBjAyBgNVHR8EKzApMCegJaAj\nhiFodHRwOi8vdDEuc3ltY2IuY29tL1RoYXd0ZVBDQS5jcmwwLwYIKwYBBQUHAQEE\nIzAhMB8GCCsGAQUFBzABhhNodHRwOi8vdDIuc3ltY2IuY29tMEEGA1UdIAQ6MDgw\nNgYKYIZIAYb4RQEHNjAoMCYGCCsGAQUFBwIBFhpodHRwczovL3d3dy50aGF3dGUu\nY29tL2NwczApBgNVHREEIjAgpB4wHDEaMBgGA1UEAxMRU3ltYW50ZWNQS0ktMS01\nMzcwHQYDVR0OBBYEFMJPSFf80U+awF04fQ4F29kutVJgMB8GA1UdIwQYMBaAFHtb\nRc+vzst6\/TGSGmq280brV0hQMA0GCSqGSIb3DQEBCwUAA4IBAQCNBt5DyXYCytkj\nl17zY9d9RMIPawr1B+WLuPrgo\/prgJK1AyzFN+DC5ZW1knAYKEKU7kt3agEPiyPs\nVk30AGnlhMji6t5bPvY8BzqUymwnscyDGmBxJ9K\/AvUeRNNI1abTdiEAnPqYZOsX\nNj\/rGzw+prHZWAYOctlovvGnINdS5KR3H3FwnVU1hTfhHU2UwnB\/lUBuS32ytCkq\nA3nIuUxnYQSgiyf\/WQDrVX\/GtzM1LV5OrLjqEsXo97mrvnSSLLfZTcqELxzC8HJ8\nsjFuz4DliAc2UXu6Ya9tjSNbNKOVvKIxf\/L157fo78S1JzLp955pxyvovrsMqufq\nYBLqJop4\n-----END CERTIFICATE-----","-----BEGIN CERTIFICATE-----\nMIIEIDCCAwigAwIBAgIQNE7VVyDV7exJ9C\/ON9srbTANBgkqhkiG9w0BAQUFADCB\nqTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYGA1UECxMf\nQ2VydGlmaWNhdGlvbiBTZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UECxMvKGMpIDIw\nMDYgdGhhd3RlLCBJbmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNlIG9ubHkxHzAdBgNV\nBAMTFnRoYXd0ZSBQcmltYXJ5IFJvb3QgQ0EwHhcNMDYxMTE3MDAwMDAwWhcNMzYw\nNzE2MjM1OTU5WjCBqTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDHRoYXd0ZSwgSW5j\nLjEoMCYGA1UECxMfQ2VydGlmaWNhdGlvbiBTZXJ2aWNlcyBEaXZpc2lvbjE4MDYG\nA1UECxMvKGMpIDIwMDYgdGhhd3RlLCBJbmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNl\nIG9ubHkxHzAdBgNVBAMTFnRoYXd0ZSBQcmltYXJ5IFJvb3QgQ0EwggEiMA0GCSqG\nSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCsoPD7gFnUnMekz52hWXMJEEUMDSxuaPFs\nW0hoSVk3\/AszGcJ3f8wQLZU0HObrTQmnHNK4yZc2AreJ1CRfBsDMRJSUjQJib+ta\n3RGNKJpchJAQeg29dGYvajig4tVUROsdB58Hum\/u6f1OCyn1PoSgAfGcq\/gcfomk\n6KHYcWUNo1F77rzSImANuVud37r8UVsLr5iy6S7pBOhih94ryNdOwUxkHt3Ph1i6\nSk\/KaAcdHJ1KxtUvkcx8cXIcxcBn6zL9yZJclNqFwJu\/U30rCfSMnZEfl2pSy94J\nNqR32HuHUETVPm4pafs5SSYeCaWAe0At6+gnhcn+Yf1+5nyXHdWdAgMBAAGjQjBA\nMA8GA1UdEwEB\/wQFMAMBAf8wDgYDVR0PAQH\/BAQDAgEGMB0GA1UdDgQWBBR7W0XP\nr87Lev0xkhpqtvNG61dIUDANBgkqhkiG9w0BAQUFAAOCAQEAeRHAS7ORtvzw6WfU\nDW5FvlXok9LOAz\/t2iWwHVfLHjp2oEzsUHboZHIMpKnxuIvW1oeEuzLlQRHAd9mz\nYJ3rG9XRbkREqaYB7FViHXe4XI5ISXycO1cRrK1zN44veFyQaEfZYGDm\/Ac9IiAX\nxPcW6cTYcvnIc3zfFi8VqT79aie2oetaupgf1eNNZAqdE8hhuvU5HIe6uL17In\/2\n\/qxAeeWsEG89jxt5dovEN7MhGITlNgDrYyCZuen+MwS7QcjBAvlEYyCegc5C09Y\/\nLHbTY5xZ3Y+m4Q6gLkH3LpVHz7z9M\/P2C2F+fpErgUfCJzDupxBdN49cOSvkBPB7\njVaMaA==\n-----END CERTIFICATE-----"],"method":"TLS","prb_id":10096,"dst_port":"443","dst_addr":"178.248.236.31","ttc":33.09869,"src_addr":"185.116.201.100","group_id":8844155,"type":"sslcert","msm_name":"SSLCert","ver":"1.2"}')
    assert(not result.is_error)
    extensions = [c.extensions for c in result.certificates]
    should_be = [
        {"subjectAltName": [u"*.tinkoff.ru", u"tinkoff.ru"]},
        {"subjectAltName": ["/CN=SymantecPKI-1-537"]},
        {},
    ]
    assert extensions == should_be


def test_invalid_country_code():
    result = Result.get('{"af":4,"cert":["-----BEGIN CERTIFICATE-----\\nMIIENTCCAx2gAwIBAgIBADANBgkqhkiG9w0BAQUFADBzMQ8wDQYDVQQGEwZCcmF6\\naWwxCTAHBgNVBAgMADESMBAGA1UEBwwJU2FvIFBhdWxvMQ4wDAYDVQQKDAVJQ0FO\\nTjEOMAwGA1UECwwFTFJPT1QxITAfBgNVBAMMGGdydTAxLmwucm9vdC1zZXJ2ZXJz\\nLm9yZzAeFw0xNzA0MDMyMTU4MzlaFw0yNzA0MDEyMTU4MzlaMHMxDzANBgNVBAYT\\nBkJyYXppbDEJMAcGA1UECAwAMRIwEAYDVQQHDAlTYW8gUGF1bG8xDjAMBgNVBAoM\\nBUlDQU5OMQ4wDAYDVQQLDAVMUk9PVDEhMB8GA1UEAwwYZ3J1MDEubC5yb290LXNl\\ncnZlcnMub3JnMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAuEHu8DoR\\nTlawVGz1bUyzFXpnnOVQwPQU9Z4q5auL74MTRizabE/6+V0Q3bfc3dLWOIY58WvX\\nq52Q81N1HynOZ5ZFYz+GMeeilxKJ4GlWQ22lYXBDobweCNhDEDQsh59AklzdyiWc\\n9g08THiaxFE85c3XlDzWCmlJGInZBZcml2VxQVEM8zrjqZXn/T3kUx8rej65q0v4\\nWEay02nrQxjUeFSRYx48WfgAz7S7LruLiPNO12pvVIpJro+MLzRrYD7f4Ba6pF8W\\nket/+nIAchbc+RjEyHyyE+hRmPfTDNARgZgSaEqB98tpav6k7Z7bijsIpwx6U4l6\\nYRxz3DIWKMe++QIDAQABo4HTMIHQMA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYE\\nFCwFtz6hwT4qsx5j0lUluDyXbJpBMIGdBgNVHSMEgZUwgZKAFCwFtz6hwT4qsx5j\\n0lUluDyXbJpBoXekdTBzMQ8wDQYDVQQGEwZCcmF6aWwxCTAHBgNVBAgMADESMBAG\\nA1UEBwwJU2FvIFBhdWxvMQ4wDAYDVQQKDAVJQ0FOTjEOMAwGA1UECwwFTFJPT1Qx\\nITAfBgNVBAMMGGdydTAxLmwucm9vdC1zZXJ2ZXJzLm9yZ4IBADANBgkqhkiG9w0B\\nAQUFAAOCAQEAVSqrE2xMOqOMT/0r0ijSJVJjMRtdd2WPnpFLZsIt5siHXZHitoF/\\nFLlgaDfWd70uVlL8EjGHxwKPJL2l8WU1foCKCHUA7syarvpQEpm3kaGAoPm8VPm8\\n/ZcjetBxvAF0ZImoAcNE66aanPRSCvcna/5ANgHyZp6a0i7DT/Tqd+/0U3PNZK0Z\\nI7K2HvWx8wgN9WnMwEcNDVw9/pvNpC4Uh4MNDXYJZeC1xMNlB3sfm/ohP15lPsxI\\nN2afNemOosSyqViiPCVl6HaEwrR7YZB2wffnMAC2RAkJYyRkv/V5L4Hf4xz3C+Bl\\n47GfJQP8dNoGhIt1iCtMQp1iRNUfYCuqhQ==\\n-----END CERTIFICATE-----"],"dst_addr":"199.7.83.80","dst_name":"199.7.83.80","dst_port":"443","from":"123.203.145.125","fw":4790,"group_id":1965073,"lts":54,"method":"TLS","msm_id":1965073,"msm_name":"SSLCert","prb_id":1004,"rt":701.811596,"src_addr":"192.168.1.100","timestamp":1502117346,"ttc":346.966342,"type":"sslcert","ver":"1.2"}')
    assert(result.is_error)
    print(type(result.error_message), result.error_message)
    assert(result.error_message == "Country name must be a 2 character country code")
